package com.bxm.adscounter.rtb.common.impl.xunfei;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;
import org.springframework.web.util.UriComponentsBuilder;

import java.nio.charset.StandardCharsets;

/**
 * @author daill
 * @date 2022-10-17
 * @since 1.0
 */
@Slf4j
public class XunFeiRtbIntegration extends AbstractClickTrackerRtbIntegration {

    private final XunFeiConfig config;

    public XunFeiRtbIntegration(XunFeiConfig config) {
        super(config);
        this.config = config;
    }

    @Override
    public String getClickIdParameterNameOnReferrer() {
        return "requestId";
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        PositionRtb positionRtb = request.getConfig();

        String clickIdParameterNameOnReferrer = getClickIdParameterNameOnReferrer();
        String requestId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst(clickIdParameterNameOnReferrer);
        if (StringUtils.isBlank(requestId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, positionRtb.getPositionId() + "Cannot found '" + clickIdParameterNameOnReferrer + "' by referrer.");
        }
        String creativeId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("creativeId");

        XunFeiRtbIntegration.Data data = new Data();
        data.setCvTime(System.currentTimeMillis());
        data.setCvType(eventType);
        data.setRequestId(requestId);
        data.setCreativeId(creativeId);

        XunFeiRequest xunFeiRequest = new XunFeiRequest();
        xunFeiRequest.setSrcId(positionRtb.getSourceId());
        xunFeiRequest.setSrcType("Web");
        xunFeiRequest.setData(data);
        String entity = JsonHelper.convert(xunFeiRequest);
        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        request.setRequestBody(entity);
        return post;
    }

    @Override
    public Rtb rtb() {
        return Rtb.XunFei;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        FeedbackResponse feedbackResponse = JsonHelper.convert(body, Response.class);
        return feedbackResponse;
    }

    private static class Response extends FeedbackResponse {

        private Integer code;
        private String message;

        @Override
        public boolean isSuccess() {
            return code == 200;
        }

        public Integer getCode() {
            return code;
        }

        public void setCode(Integer code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }


    private static class XunFeiRequest {
        private String srcType;
        private XunFeiRtbIntegration.Data data;
        private String srcId;

        public String getSrcType() {
            return srcType;
        }

        public void setSrcType(String srcType) {
            this.srcType = srcType;
        }

        public XunFeiRtbIntegration.Data getData() {
            return data;
        }

        public void setData(XunFeiRtbIntegration.Data data) {
            this.data = data;
        }

        public String getSrcId() {
            return srcId;
        }

        public void setSrcId(String srcId) {
            this.srcId = srcId;
        }
    }


    private static class Data {
        private String cvType;
        private Long cvTime;
        private String requestId;
        private String creativeId;

        public String getCvType() {
            return cvType;
        }

        public void setCvType(String cvType) {
            this.cvType = cvType;
        }

        public Long getCvTime() {
            return cvTime;
        }

        public void setCvTime(Long cvTime) {
            this.cvTime = cvTime;
        }

        public String getRequestId() {
            return requestId;
        }

        public String getCreativeId() {
            return creativeId;
        }

        public void setCreativeId(String creativeId) {
            this.creativeId = creativeId;
        }

        public void setRequestId(String requestId) {
            this.requestId = requestId;
        }
    }

}
