package com.bxm.adscounter.rtb.common.impl.yixiao;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Optional;

/**
 * @author daill
 * @date 2022-08-22
 * @since 1.0
 */
@Slf4j
public class YiXiaoRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "callback";

    public YiXiaoRtbIntegration(YiXiaoConfig config) {
        super(config);
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String ref = clickEventLog.getRef();
        String clickId = UrlHelper.getFirstValueOfParamName(ref, CLICK_ID);
        if (StringUtils.isBlank(clickId)) {
            log.warn("Not found: {}", ref);
            return null;
        }
        return DigestUtils.md5Hex(UrlHelper.urlDecode(clickId));
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String clickId = UrlHelper.getFirstValueOfParamName(request.getReferrer(), CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(DigestUtils.md5Hex(UrlHelper.urlDecode(clickId)));

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String referrer = request.getReferrer();
        PositionRtb positionRtb = request.getConfig();
        String eventType = request.getEventType();

        String callback = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("callback");
        if (StringUtils.isBlank(callback)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, positionRtb.getPositionId() + "： Cannot found 'callback' by referrer.");
        }
        String url = UrlHelper.urlDecode(callback) +"&conv_type="+eventType+"&t="+System.currentTimeMillis();
        return new HttpGet(url);
    }

    @Override
    public Rtb rtb() {
        return Rtb.YiXiao;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }


    private static class Response extends FeedbackResponse {
        private String status;
        private String msg;

        @Override
        public boolean isSuccess() {
            return StringUtils.equalsIgnoreCase("SUCCESSED", status);
        }

        public String getStatus() {
            return status;
        }

        public void setStatus(String status) {
            this.status = status;
        }

        public String getMsg() {
            return msg;
        }

        public void setMsg(String msg) {
            this.msg = msg;
        }
    }

}
