package com.bxm.adscounter.rtb.common.service;

import com.bxm.adscounter.rtb.common.*;
import com.bxm.adscounter.rtb.common.control.ControlUtils;
import com.bxm.adscounter.rtb.common.control.RateControlRtbIntegration;
import com.bxm.adscounter.rtb.common.control.cpa.CpaControl;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adscounter.rtb.common.utils.InadsExtValueUtils;
import com.bxm.adscounter.rtb.common.utils.PositionRTBUtils;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * @author allen
 * @date 2022-05-13
 * @since 1.0
 */
@Slf4j
public abstract class AbstractFeedbackService implements FeedbackService {

    private Fetcher fetcher;
    private Counter counter;
    private RtbIntegrationFactory factory;
    private PositionRtbService service;
    private CpaControl cpaControl;

    private static final int ACTIVITY_FEEDBACK_LIMIT = 5;

    @Autowired
    public void setFetcher(Fetcher fetcher) {
        this.fetcher = fetcher;
    }

    @Autowired
    public void setFactory(RtbIntegrationFactory factory) {
        this.factory = factory;
    }

    @Autowired
    public void setService(PositionRtbService service) {
        this.service = service;
    }

    @Autowired
    public void setCounter(Counter counter) {
        this.counter = counter;
    }

    @Autowired
    public void setCpaControl(CpaControl cpaControl) {
        this.cpaControl = cpaControl;
    }

    /**
     * 根据指定配置来决定是否需要进行回传。
     * @param config 配置
     * @return true 则需要回传。
     */
    protected abstract boolean doFeedbackIfNecessary(PositionRtb config);

    @Override
    public boolean doFeedback(KeyValueMap keyValueMap) throws RtbIntegrationException {
        return doFeedback(keyValueMap, null);
    }

    @Override
    public boolean doFeedback(KeyValueMap keyValueMap, Updating updating) throws RtbIntegrationException {
        String bxmId = keyValueMap.getFirst(Inads.Param.BXMID);
        String positionId = keyValueMap.getFirst(Inads.Param.TAGID);

        if (StringUtils.isBlank(positionId)) {
            log.warn("[{}] Cannot found tagid!", bxmId);
            return true;
        }
        Rtb rtb = getRtb();
        RtbIntegration integration = factory.get(rtb);
        if (Objects.isNull(integration)) {
            log.warn("Not found RtbIntegration for {}", rtb);
            return true;
        }

        PositionRtb config = fetchPositionRtbConfig(positionId);
        if (Objects.isNull(config)) {
            return true;
        }

        String mt = keyValueMap.getFirst(Inads.Param.MT);
        String uid = keyValueMap.getFirst(Inads.Param.UID);

        boolean isActivityAttend = StringUtils.equalsIgnoreCase(mt, Inads.Mt.ActivityAttend.original());
        boolean isAdClick = StringUtils.equalsIgnoreCase(mt, Inads.Mt.AdClick.original());
        boolean isAdConversion = StringUtils.equalsIgnoreCase(mt, Inads.Mt.AdConversion.original());

        int times = isActivityAttend ? fetchActivityAttendTimes(uid) : 0;

        // 正常回传
        {
            if (doFeedbackIfNecessary(config)) {
                if (isActivityAttend) {
                    // 活动参与
                    return doFeedbackOnActivityAttend(keyValueMap, updating, config, times);
                } else if (isAdClick && isFeedbackIfSkipCpaOrSequenceLimit(keyValueMap, config)) {
                    // 广告点击
                    return doFeedbackOnOther(keyValueMap, NumberUtils.toInt(PositionRTBUtils.AD_CLICK), updating, config,  (request, rtbIntegration) -> true);
                } else if (isAdConversion && isFeedbackIfSkipCpaOrSequenceLimit(keyValueMap, config)) {
                    // 广告回传
                    // 设置一个拦截器，供cpa累加模式回传判断使用
                    BiFunction<FeedbackRequest, RtbIntegration, Boolean> feedbackInterceptor =
                            (request, rtbIntegration) -> cpaControl.isFeedback(request, rtbIntegration);
                    String convType = keyValueMap.getFirst(Inads.Param.CONVERSION_TYPE);
                    int type = NumberUtils.toInt(convType, 0);
                    return doFeedbackOnOther(keyValueMap, type, updating, config, feedbackInterceptor);
                }
            }
        }

        return true;
    }

    /**
     * 判断当前广告位上的回传次序、CPA限额的检查条件是否通过。
     * @param keyValueMap KeyValueMap
     * @param config 配置
     * @return 当返回 true 则表示通过了条件检查，需要回传。否则不允许回传。如果发生异常，也会返回 true
     */
    private boolean isFeedbackIfSkipCpaOrSequenceLimit(KeyValueMap keyValueMap, PositionRtb config) {
        try {
            // 不是头部券模式，不用判断后续逻辑
            if (!config.isFeedbackModeHeadTicket()) {
                return true;
            }

            String ticketId = keyValueMap.getFirst(Inads.Param.ADID);
            String positionId = keyValueMap.getFirst(Inads.Param.TAGID);

            Set<String> sequence = Sets.newHashSet(Optional.ofNullable(config.getUserActSeqList()).orElse(new ArrayList<>()));
            Integer maxCpaLimit = config.getCpa();

            // 活动发券次序
            String rank = getAcquiredRankByActivity(keyValueMap);
            if (StringUtils.isNotBlank(rank) && CollectionUtils.isNotEmpty(sequence) && !sequence.contains(rank)) {
                return false;
            }

            if (Objects.isNull(maxCpaLimit)) {
                return true;
            }
            // 头部券不处理CPA限制
            if (PositionRTBUtils.isHeadTicket(ticketId, config)) {
                return true;
            }
            // 限制cpa
            Integer configCpaPrice = getConfigCpaPrice(positionId, ticketId);
            // 为空代表cpc全或者cpa券（不处理限制cpa）
            if (Objects.isNull(configCpaPrice)) {
                return true;
            }
            // 如果配置的价格大于限制的，则允许回传
            return configCpaPrice > maxCpaLimit;
        } catch (Exception e) {
            log.error("isFeedbackIfSkipCpaOrSequenceLimit: ", e);
            return true;
        }
    }

    /**
     * 获取广告券的RTB cpa
     */
    private Integer getConfigCpaPrice(String positionId, String ticketId) {
        Integer price = fetcher.hfetch(new TargetFactory<Integer>()
                .keyGenerator(TicketKeyGenerator.getTicketRTBCPAPrice(ticketId))
                .field(positionId)
                .skipNativeCache(true)
                .cls(Integer.class)
                .build());
        if (Objects.nonNull(price)) {
            return price;
        }
        price = fetcher.hfetch(new TargetFactory<Integer>()
                .keyGenerator(TicketKeyGenerator.getTicketRTBCPAPrice(ticketId))
                .field(ticketId)
                .skipNativeCache(true)
                .cls(Integer.class)
                .build());
        return price;
    }

    private boolean doFeedbackOnActivityAttend(KeyValueMap keyValueMap, Updating updating, PositionRtb config, int times) throws RtbIntegrationException {
        // RTB 浅层目标
        String targetOneRtb = config.getTargetOneRtb();
        // RTB 深层目标
        String targetTwoRtb = config.getTargetTwoRtb();
        // 活动浅层
        String activityOne = config.getActivityOne();
        // 活动深层
        String activityTwo = config.getActivityTwo();


        String shallowKey = ControlUtils.createKey(config.getPositionId(), "shallow");
        String deepKey = ControlUtils.createKey(config.getPositionId(), "deep");

        //活动限制：如果浅层和深层同时限制，那么当浅层和深层都达到限制才不回传。
        boolean allowFeedbackShallow = fetchActivityFeedbackTimes(shallowKey) < ACTIVITY_FEEDBACK_LIMIT;
        boolean allowFeedbackDeep = fetchActivityFeedbackTimes(deepKey) < ACTIVITY_FEEDBACK_LIMIT;

        // 活动浅层 配置的是 活动参与
        if (StringUtils.equals(activityOne, PositionRTBUtils.ACTIVITY_ATTEND)) {
            if (config.isEnableActivityFeedbackLimitShallow()) {
                if (allowFeedbackShallow) {
                    boolean feedbackSuccess = doFeedback0(keyValueMap, updating, config, targetOneRtb, FeedbackRequest.SHALLOW_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_ATTEND);
                    if (feedbackSuccess) {
                        counter.incrementAndGet(stringActivityFeedbackTimes(shallowKey));
                    }
                    return feedbackSuccess;
                }
            } else {
                return doFeedback0(keyValueMap, updating, config, targetOneRtb, FeedbackRequest.SHALLOW_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_ATTEND);
            }
        }
        // 活动深层 配置的是 活动参与
        if (StringUtils.equals(activityTwo, PositionRTBUtils.ACTIVITY_ATTEND)) {
            if (config.isEnableActivityFeedbackLimitDeep()) {
                if (allowFeedbackDeep) {
                    boolean feedbackSuccess = doFeedback0(keyValueMap, updating, config, targetTwoRtb, FeedbackRequest.DEEP_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_ATTEND);
                    if (feedbackSuccess) {
                        counter.incrementAndGet(stringActivityFeedbackTimes(deepKey));
                    }
                    return feedbackSuccess;
                }
            } else {
                return doFeedback0(keyValueMap, updating, config, targetTwoRtb, FeedbackRequest.DEEP_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_ATTEND);
            }
        }

        // 如果 活动浅层 或 活动深层，配置的是活动复参，那么回传
        if (StringUtils.equals(activityOne, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND) || StringUtils.equals(activityTwo, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND)) {
            if (StringUtils.equals(activityOne, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND) && times == 2) {
                return doFeedback0(keyValueMap, updating, config, targetOneRtb, FeedbackRequest.SHALLOW_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND);
            }
            if (StringUtils.equals(activityTwo, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND) && times == 2) {
                return doFeedback0(keyValueMap, updating, config, targetTwoRtb, FeedbackRequest.DEEP_CONVERSION_LEVEL, PositionRTBUtils.ACTIVITY_REPEAT_ATTEND);
            }
        }
        return true;
    }

    private boolean doFeedbackOnOther(KeyValueMap keyValueMap, int type, Updating updating, PositionRtb config, BiFunction<FeedbackRequest, RtbIntegration, Boolean> feedbackInterceptor) throws RtbIntegrationException {
        String adid = keyValueMap.getFirst(Inads.Param.ADID);
        // RTB 浅层目标
        String targetOneRtb = config.getTargetOneRtb();
        // RTB 深层目标
        String targetTwoRtb = config.getTargetTwoRtb();

        // 判断互动累加回传的情况。这里需放置于头部券逻辑之前
        if (config.isFeedbackModeSummaryCpa()) {
            // 累加模式下的券点击不回传
            if (PositionRTBUtils.isAdClick(type) || Objects.isNull(targetOneRtb)) {
                return true;
            }
            return doFeedback0(keyValueMap, updating, config, targetOneRtb, FeedbackRequest.SHALLOW_CONVERSION_LEVEL, TypeHelper.castToString(type), feedbackInterceptor);
        }

        PositionRtb.TargetBXM target = PositionRTBUtils.convert(adid, config);
        if (Objects.isNull(target)) {
            return true;
        }
        Map<String, Object> extMap = getExt(keyValueMap);

        int targetOneBxm = Optional.ofNullable(target.getTargetOneBxm()).orElse(-1);
        if (targetOneBxm == NumberUtils.toInt(PositionRTBUtils.AD_TYPE)) {
            targetOneBxm = Optional.ofNullable(TypeHelper.castToInt(extMap.get(Inads.Param.SHALLOW_TYPE))).orElse(-1);
            if (targetOneBxm == -1) {
                log.error("[{}] bxmId Cannot found targetOneBxm ! adid: {}", keyValueMap.getFirst(Inads.Param.BXMID),adid);
            }
        }

        if (type == targetOneBxm && Objects.nonNull(targetOneRtb)) {
            return doFeedback0(keyValueMap, updating, config, targetOneRtb, FeedbackRequest.SHALLOW_CONVERSION_LEVEL, TypeHelper.castToString(type), feedbackInterceptor);
        }

        int targetTwoBxm = Optional.ofNullable(target.getTargetTwoBxm()).orElse(-1);
        if (targetTwoBxm == NumberUtils.toInt(PositionRTBUtils.AD_TYPE)) {
            targetTwoBxm = Optional.ofNullable(TypeHelper.castToInt(extMap.get(Inads.Param.DEEP_TYPE))).orElse(-1);
            if (targetTwoBxm == -1) {
                log.error("[{}] bxmId Cannot found targetTwoBxm! adid: {}", keyValueMap.getFirst(Inads.Param.BXMID) ,adid);
            }
        }

        if (type == targetTwoBxm && Objects.nonNull(targetTwoRtb)) {
            return doFeedback0(keyValueMap, updating, config, targetTwoRtb, FeedbackRequest.DEEP_CONVERSION_LEVEL, TypeHelper.castToString(type), feedbackInterceptor);
        }
        return true;
    }

    private boolean doFeedback0(KeyValueMap keyValueMap, Updating updating, PositionRtb config,
                                String eventType, int conversionLevel, String conversionType) throws RtbIntegrationException {
        return doFeedback0(keyValueMap, updating, config, eventType, conversionLevel, conversionType, (request, integration) -> true);
    }

    private boolean doFeedback0(KeyValueMap keyValueMap, Updating updating, PositionRtb config,
                                String eventType, int conversionLevel, String conversionType,
                                BiFunction<FeedbackRequest, RtbIntegration, Boolean> feedbackInterceptor) throws RtbIntegrationException {
        Rtb rtb = getRtb();
        RtbIntegration integration = factory.get(rtb);
        if (Objects.isNull(integration)) {
            log.warn("Not found RtbIntegration for {}", rtb);
            return true;
        }
        FeedbackRequest request = createFeedbackRequest(integration, keyValueMap, updating, config, conversionLevel, conversionType, eventType);

        // 判断拦截器逻辑，然后回传
        if (Objects.isNull(feedbackInterceptor)) {
            return integration.doFeedback(request).isSuccess();
        }
        if (feedbackInterceptor.apply(request, integration)) {
            return integration.doFeedback(request).isSuccess();
        }
        return true;
    }

    /**
     * 创建请求
     *
     * @param integration 集成实现
     * @param keyValueMap KeyValueMap
     * @param updating 更新器
     * @param config 配置
     * @param conversionLevel 转化级别
     * @param conversionType 变现猫收到的转化类型
     * @param eventType 回传给RTB平台的转化类型
     * @return 请求对象
     */
    private FeedbackRequest createFeedbackRequest(RtbIntegration integration, KeyValueMap keyValueMap, Updating updating, PositionRtb config,
                                                  int conversionLevel,
                                                  String conversionType, String eventType) {
        String referrer = keyValueMap.getFirst(Inads.Param.REFER);
        FeedbackRequest request = FeedbackRequest.builder()
                .config(config)
                .conversionLevel(conversionLevel)
                .conversionType(conversionType)
                .keyValueMap(keyValueMap)
                .eventType(eventType)
                .referrer(referrer)
                .smartConvType(SmartConvType.NONE)
                .build();
        if (Objects.nonNull(updating)) {
            updating.accept(request, config);
        }
        ActionType actionType = ActionType.of(keyValueMap, config);
        if (Objects.nonNull(actionType)) {
            request.setActionType(actionType);
        }
        // setAdGroupId 要放在 setActionType 之后，因为 fetchAdGroupId 有些实现可能需要 ActionType。
        request.setAdGroupId(fetchAdGroupId(integration, request));
        return request;
    }

    private String fetchAdGroupId(RtbIntegration integration, FeedbackRequest request) {
        String adGroupId = null;
        if (integration instanceof ClickTracker) {
            ClickTracker clickTracker = (ClickTracker) integration;
            adGroupId = clickTracker.getAdGroupId(request);
            adGroupId = clickTracker.fixAdGroupIdIfInvalid(adGroupId);
        }
        return adGroupId;
    }

    private PositionRtb fetchPositionRtbConfig(String positionId) {
        return service.get(positionId);
    }

    private int fetchActivityAttendTimes(String uid) {
        return Optional.ofNullable(fetcher.fetch(new TargetFactory<Integer>()
                .keyGenerator(PositionRTBUtils.stringActivityAttend(uid))
                .skipNativeCache(true)
                .cls(Integer.class)
                .build()))
                .orElse(0);
    }

    private int fetchActivityFeedbackTimes(String dimensionKey) {
        return Optional.ofNullable(fetcher.fetch(new TargetFactory<Integer>()
                .keyGenerator(stringActivityFeedbackTimes(dimensionKey))
                .skipNativeCache(true)
                .cls(Integer.class)
                .build()))
                .orElse(0);
    }

    public static KeyGenerator stringActivityFeedbackTimes(String dimensionKey) {
        return () -> KeyBuilder.build("RTB", "ACTIVITY_LIMIT", "TIMES", dimensionKey);
    }

    public static String getAcquiredRankByActivity(KeyValueMap keyValueMap) {
        String ext = keyValueMap.getFirst(Inads.Param.EXT);
        if (StringUtils.isBlank(ext)) {
            return null;
        }
        Map<String, Object> map = InadsExtValueUtils.parse(ext);
        return Optional.ofNullable(map).map(e -> TypeHelper.castToString(e.get("userActSeq"))).orElse(null);
    }

    public static Map<String, Object> getExt(KeyValueMap keyValueMap) {
        String ext = keyValueMap.getFirst(Inads.Param.EXT);
        if (StringUtils.isBlank(ext)) {
            return new HashMap<>();
        }
        Map<String, Object> map = InadsExtValueUtils.parse(ext);
        return map;
    }

}
