package com.bxm.adsprod.api.controller;

import com.bxm.adscounter.facade.Mt;
import com.bxm.adscounter.facade.model.NoStatisticsException;
import com.bxm.adscounter.facade.model.Response;
import com.bxm.adscounter.facade.model.TicketCountMsgDto;
import com.bxm.adscounter.facade.model.TicketCounterRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.TicketService;
import com.bxm.adsprod.integration.adscounter.AdscounterServiceIntegration;
import com.bxm.warcar.utils.IpHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;

/**
 * @author allen
 * @date 2019/6/4
 * @since 1.0.0
 */
@RestController
@RequestMapping("/limit")
public class LimitController {

    private static final Logger LOGGER = LoggerFactory.getLogger(LimitController.class);
    private final ThreadPoolExecutor dotThreadPool = new ThreadPoolExecutor(10, 10, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("dot"));

    @Resource(name = "redisFetcher")
    private Fetcher fetcher;
    @Autowired
    private TicketService ticketService;
    @Autowired
    private AdscounterServiceIntegration asi;

    @RequestMapping("/fetch")
    public String[] fetch(HttpServletRequest request) {
        Map<String, String> enableds = fetcher.hfetchall(new TargetFactory<String>()
                .keyGenerator(TicketKeyGenerator.Threshold.getLimitPriceEnabled())
                .cls(String.class)
                .build());

        if (MapUtils.isEmpty(enableds)) {
            return new String[0];
        }

        List<Ticket> tickets = Lists.newArrayList();
        Set<String> ticketIds = enableds.keySet();
        for (String ticketId : ticketIds) {
            Ticket ticket = ticketService.get(new BigInteger(ticketId));
            if (null != ticket && ticket.isAvailableForStatus()) {
                tickets.add(ticket);
            }
        }
        if (CollectionUtils.isEmpty(tickets)) {
            return new String[0];
        }

        Ticket ticket = tickets.get(RandomUtils.nextInt(tickets.size()));
        String ticketId = String.valueOf(ticket.getId());
        String values = enableds.get(ticketId);
        if (StringUtils.isBlank(values)) {
            return new String[0];
        }
        String[] item = StringUtils.split(values, "-");
        double avgClickPrice = NumberUtils.toDouble(item[0]);
        double clickRatio = NumberUtils.toDouble(item[1]);
        if (avgClickPrice == 0 || clickRatio == 0) {
            return new String[0];
        }

        BigDecimal times = new BigDecimal(1).divide(new BigDecimal(clickRatio), 2, BigDecimal.ROUND_HALF_UP);
        int currentTimes = times.intValue();
        double currentRemainder = times.subtract(new BigDecimal(currentTimes)).doubleValue();
        int remainderTimes = getRemainderTimes(ticketId, currentRemainder);
        int totalTimes = currentTimes + remainderTimes;

        for (int i = 0; i < totalTimes; i++) {
            dotThreadPool.submit(new DotSimulation(asi, create(request, ticket), Mt._5.getOriginal()));
            dotThreadPool.submit(new DotSimulation(asi, create(request, ticket), Mt._6.getOriginal()));
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("[{}] avgClickPrice={}, clickRatio={}. originDotTime: {}, totalTimes: {}, remainderTimes: {}, dotRatio: {}",
                    ticketId, avgClickPrice, clickRatio, times, totalTimes, currentRemainder,
                    (new BigDecimal(1).divide(new BigDecimal(totalTimes), 3, BigDecimal.ROUND_HALF_UP).doubleValue()));
        }

        Future<Response> future = dotThreadPool.submit(new DotSimulation(asi, create(request, ticket), Mt._7.getOriginal()));

        String bxmId = null;
        try {
            Response response = future.get();
            if (null != response) {
                bxmId = response.getOrderId();
            }
        } catch (InterruptedException | ExecutionException e) {
            LOGGER.error("get: ", e);
        }

        String url = ticket.getUrl();
        if (StringUtils.isNotBlank(bxmId)) {
            url += StringUtils.contains(url, "?") ? "&bxm_id=" + bxmId : "?bxm_id=" + bxmId;
        }
        return new String[] {url};
    }

    private int getRemainderTimes(String ticketId, double currentRemainder) {
        JedisPool jedisPool = fetcher.getClientOriginal();
        Jedis jedis = jedisPool.getResource();
        try {
            int times = 0;
            double remainder = jedis.hincrByFloat(TicketKeyGenerator.Threshold.getLimitPriceRemainder().generateKey(), ticketId, currentRemainder);
            while (remainder >= 1) {
                remainder = jedis.hincrByFloat(TicketKeyGenerator.Threshold.getLimitPriceRemainder().generateKey(), ticketId, -1);
                if (remainder >= 0) {
                    times++;
                }
            }
            return times;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    private TicketCountMsgDto create(HttpServletRequest request, Ticket ticket) {
        TicketCountMsgDto dto = new TicketCountMsgDto();
        dto.setAppkey("fbca14bb5bb74d5a9b5fa165ad78a15a");
        dto.setBusiness("money-20");
        dto.setReferrer(request.getHeader("Referer"));
        dto.setUserAgent(request.getHeader("User-Agent"));
        dto.setActivityid(12651L);
        dto.setAppos(1);
        dto.setUa("0");
        dto.setPreid(ticket.getId().longValue());
        dto.setIpAddress(IpHelper.getIpFromHeader(request));
        dto.setAwardtype(2);
        dto.setAssetsId("-1");
        dto.setUid("_____DEFAULT_UID_FOR_LIMIT_PRICE");
        return dto;
    }

    static class DotSimulation implements Callable<Response> {

        private static final Logger LOGGER = LoggerFactory.getLogger(DotSimulation.class);

        private final AdscounterServiceIntegration asi;
        private final TicketCountMsgDto dto;
        private final int modelType;

        public DotSimulation(AdscounterServiceIntegration asi, TicketCountMsgDto dto, int modelType) {
            this.asi = asi;
            this.dto = dto;
            this.modelType = modelType;
        }

        @Override
        public Response call() {
            try {
                TicketCounterRequest tcr = new TicketCounterRequest();
                dto.setModeltype(modelType);
                tcr.setTicketCountMsgDto(dto);
                return asi.saveTicketCountMsg2(tcr);
            } catch (NoStatisticsException e) {
                LOGGER.error("saveTicketCountMsg2: ", e);
                return null;
            }
        }
    }
}
