package com.bxm.adsprod.counter.ticket;

import com.bxm.adsprod.common.message.AbstractMessageListener;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.TypeHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigInteger;

/**
 * <h3>抽象的统计器</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
public abstract class AbstractCounter<REQUEST, RETURNING> extends AbstractMessageListener {

    protected static final long DEFAULT_INCREMENT_VALUE = 1;

    @Autowired
    @Qualifier("jedisCounter")
    protected Counter counter;

    /**
     * 转换Request对象
     * @param request
     * @return
     */
    protected abstract REQUEST convertRequest(Object request);

    /**
     * 转换Returning对象
     * @param returning
     * @return
     */
    protected abstract RETURNING convertReturning(Object returning);

    /**
     * 会首先从{@link #getKeyGenerator(Object, Object)}获取，如果返回为<code>null</code> 则使用这个返回存储需要的KeyGenerator实现
     * @param uid
     * @param ticketId
     * @return
     * @see #getKeyGenerator(Object, Object)
     */
    protected abstract KeyGenerator getKeyGenerator(String uid, BigInteger ticketId);

    /**
     * 返回存储需要的KeyGenerator实现，可以覆盖这个方法。
     * @param request
     * @param returning
     * @return
     * @see #getKeyGenerator(String, BigInteger)
     */
    protected KeyGenerator getKeyGenerator(REQUEST request, RETURNING returning) {
        return null;
    }

    /**
     * 返回增量值
     * @param request
     * @param returning
     * @return
     */
    protected abstract long getIncrementValue(REQUEST request, RETURNING returning);

    /**
     * 返回UID
     * @param request
     * @param returning
     * @return
     */
    protected abstract String getUid(REQUEST request, RETURNING returning);

    /**
     * 返回广告券ID
     * @param request
     * @param returning
     * @return
     */
    protected abstract BigInteger getTicketId(REQUEST request, RETURNING returning);

    /**
     * 返回过期时间，默认为10天
     * @return
     */
    protected int getExpireTimeInSeconds() {
        return TypeHelper.castToInt(DateHelper.getRemainSecondsOfDay(10));
    }

    /**
     * 处理完成之前需要执行的自定义方法
     * @param val 增量的值
     * @param request
     * @param returning
     */
    protected void beforeIncrement(long val, REQUEST request, RETURNING returning) {}

    /**
     * 处理完成之后需要执行的自定义方法
     * @param val 处理完成之后的值
     * @param request
     * @param returning
     */
    protected void afterIncrement(long val, REQUEST request, RETURNING returning) {}

    /**
     * 是否需要执行increment
     * @description
     * @author 阿导
     * @time 2018/3/29
     * @CopyRight 杭州微财科技有限公司
     * @param request
     * @param returning
     * @return boolean
     */
    protected boolean isIncrementIfNeeded(REQUEST request, RETURNING returning) {
        return true;
    }

    @Override
    protected final void consume(Message message, Object request, Object returning) {
        REQUEST r = convertRequest(request);
        RETURNING rtn = convertReturning(returning);
        long incrementValue = getIncrementValue(r, rtn);
        String uid = getUid(r, rtn);
        BigInteger ticketId = getTicketId(r, rtn);
        beforeIncrement(incrementValue, r, rtn);

        Long rst = 0L;
        if (isIncrementIfNeeded(r, rtn)) {
            KeyGenerator keyGenerator = getKeyGenerator(r, rtn);
            if (null == keyGenerator) {
                keyGenerator = getKeyGenerator(uid, ticketId);
            }
            rst = increment(keyGenerator, uid, ticketId, incrementValue);
        }
        afterIncrement(rst, r, rtn);
    }

    private Long increment(KeyGenerator keyGenerator, String uid, BigInteger ticketId, long inc) {
        if (this instanceof HashCounter) {
            String field = ((HashCounter) this).getField(uid, ticketId);
            return counter.hincrementByAndGet(keyGenerator, field, inc, getExpireTimeInSeconds());
        }
        else {
            return counter.incrementByAndGet(keyGenerator, inc, getExpireTimeInSeconds());
        }
    }
}
