package com.bxm.adsprod.counter.ticket.cashier;

import com.bxm.adsprod.common.eventbus.EventPark;
import com.bxm.adsprod.common.message.AbstractMessageListener;
import com.bxm.adsprod.counter.properties.Configuration;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.autoconfigure.alions.AlionsConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import java.math.BigInteger;

/**
 * <h3>广告主收银员，每一次点击扣款一次</h3>
 * <p>该实现需要结合{@link com.bxm.adsprod.facade.advertiser.AdvertiserService}使用</p>
 *
 * @see com.bxm.adsprod.facade.advertiser.AdvertiserService#incrementBalance(BigInteger, Long)
 * @author allen
 * @since V1.0.0 2017/12/15
 */
@EnableConfigurationProperties({AlionsConfiguration.class, Configuration.class})
public abstract class AdvertiserCashier extends AbstractMessageListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdvertiserCashier.class);
    @Autowired
    @Qualifier("jedisCounter")
    private Counter counter;
    @Autowired
    private AlionsConfiguration alionsConfiguration;
    @Autowired
    private Configuration local;
    @Autowired
    private EventPark eventPark;

    protected abstract void afterConsume(ClickRequest request, Ticket ticket, long inc, long afterInc);

    private long getIncrementValue(ClickRequest request, Ticket ticket) {
        if (request.isValid()) {
            return 0;
        }
        Integer offerPrice = ticket.getOfferPrice();
        return - (ticket.isCPC() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);
    }

    @Override
    protected void consume(Message message, Object request, Object returning) {
        consume((ClickRequest) request, (Ticket) returning);
    }

    private void consume(ClickRequest request, Ticket ticket) {
        long incrementValue = getIncrementValue(request, ticket);
        long remain = increment(request.getUid(), ticket, incrementValue);
        try {
            afterConsume(request, ticket, incrementValue, remain);
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("afterConsume: ", e);
            }
        }
    }

    private long increment(String uid, Ticket ticket, long inc) {
        KeyGenerator keyGenerator = TicketKeyGenerator.Advertiser.getBalance();
        return counter.hincrementByAndGet(keyGenerator, String.valueOf(ticket.getAdvertiser()), inc);
    }

    @Override
    public String getTopic() {
        return local.getTopic().getClick();
    }

}
