package com.bxm.adsprod.counter.ticket.counter;

import com.alibaba.dubbo.config.annotation.Reference;
import com.bxm.adsprod.common.eventbus.EventPark;
import com.bxm.adsprod.counter.event.BudgetNotEnoughOfDailyEvent;
import com.bxm.adsprod.counter.ticket.AbstractClickCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.media.PositionGroupService;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.ViewRequest;
import com.bxm.warcar.cache.KeyGenerator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigInteger;

/**
 * <h3>CPC广告券日预算统计器</h3>
 * <p>
 *     CPC 每一次预算触发是：点击，因此它消费的是广告券点击主题，并且在内部需要对CPA 的广告券计数增量修改为：0
 * </p>
 * <p>当日预算不足时，会对该广告券的状态更新为暂停</p>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
@Component
public class TicketForCpcDailyBudgetCounter extends AbstractClickCounter implements HashCounter {

    @Autowired
    private EventPark eventPark;
    @Reference(version = "1.0.0")
    private PositionGroupService positionGroupService;

    @Override
    public String getField(String uid, BigInteger ticketId) {
        return String.valueOf(ticketId);
    }

    @Override
    protected long getIncrementValue(ClickRequest clickRequest, Ticket ticket) {
        if (clickRequest.isValid()) {
            return 0;
        }
        Integer offerPrice = ticket.getOfferPrice();
        return (ticket.isCPC() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);
    }

    /**
     * @param val          增量的值
     * @param clickRequest
     * @param ticket
     * @see TicketForCpaDailyBudgetCounter#beforeIncrement(long, ViewRequest, Ticket)
     */
    @Override
    protected void beforeIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        String position = clickRequest.getPosition();
        BigInteger ticketId = ticket.getId();
        positionGroupService.incrementIfNecessary(ticketId, position, val);
    }

    @Override
    protected void afterIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        // val = budgetOfToday
        if (val >= ticket.getBudgetDaily()) {
            // Post message to event park.
            eventPark.post(new BudgetNotEnoughOfDailyEvent(this, ticket));
        }
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getBudgetOfDaily();
    }

}
