package com.bxm.adsprod.counter.ticket.counter;

import com.alibaba.dubbo.config.annotation.Reference;
import com.aliyun.openservices.shade.io.netty.util.internal.ConcurrentSet;
import com.bxm.adsprod.common.message.AbstractMessageListener;
import com.bxm.adsprod.counter.properties.Configuration;
import com.bxm.adsprod.counter.ticket.counter.entity.OcpcClickHistory;
import com.bxm.adsprod.counter.ticket.counter.entity.OcpcValidClickHistory;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.OcpcService;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.warcar.cache.Lock;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.GroupOperation;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * @author allen
 * @since 1.0.0
 */
@Component
public class TicketOcpcClickCounter extends AbstractMessageListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketOcpcClickCounter.class);
    private static final String COLLECTION_NAME_PREFIX = "ocpc";
    private static final int MIN_SIZE = 100;

    private static final ConcurrentSet<String> COLLECTIONNAME_CACHE = new ConcurrentSet<>();

    private final MongoTemplate mongoTemplate;
    private final Configuration configuration;

    @Reference(version = "1.0.0", timeout = 60000)
    private OcpcService ocpcService;
    @Autowired
    @Qualifier("jedisLock")
    private Lock lock;

    public TicketOcpcClickCounter(MongoTemplate mongoTemplate, Configuration configuration) {
        this.mongoTemplate = mongoTemplate;
        this.configuration = configuration;
    }

    @Override
    protected void consume(Message message, Object request, Object returning) {
        this.consume(message, (ClickRequest) request, (Ticket) returning);
    }

    private void consume(Message message, ClickRequest request, Ticket returning) {
        if (!returning.isOcpc()) {
            return;
        }

        String position = request.getPosition();
        BigInteger ticketId = request.getTicketId();
        boolean valid = request.isValid();
        Integer offerPrice = returning.getOfferPrice();

        String collectionName = createCollectionName(position, ticketId);
        String validClickCollectionName = "valid_click_details";

        if (!COLLECTIONNAME_CACHE.contains(collectionName)) {
            createCollectionIfNecess(collectionName);
        }
        if (!COLLECTIONNAME_CACHE.contains(validClickCollectionName)) {
            createCollectionIfNecess(validClickCollectionName);
        }

        if (valid) {
            long start = System.currentTimeMillis();

            String billid = request.getBillid();
            OcpcValidClickHistory o = new OcpcValidClickHistory();
            o.setBillid(billid);
            o.setPosition(position);
            o.setTicketid(TypeHelper.castToLong(ticketId));
            o.setTime(new Date());
            mongoTemplate.insert(o, validClickCollectionName);

            Count clickStat = aggregate(collectionName, 24);
            Count validStat = aggregate(validClickCollectionName, position, TypeHelper.castToLong(ticketId), 24);

            // 计算有效点击，查询可变的时间段数据
            long consume = clickStat.getConsume(), validClickCount = validStat.getValidClickCount();
            ocpcService.doValidClick(returning, position, consume, validClickCount);
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Process click of valid data {}, {} in {} ms", validClickCount, consume, (System.currentTimeMillis() - start));
            }
        } else {
            String billid = returning.getBillid();
            OcpcClickHistory o = new OcpcClickHistory();
            o.setBillid(billid);
            o.setPosition(position);
            o.setTicketid(ticketId);
            o.setPrice(offerPrice);
            o.setTime(returning.getTime());
            mongoTemplate.insert(o, collectionName);
        }
    }

    private Count aggregate(String collectionName, int lastHours) {
        long start = System.currentTimeMillis();
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, calendar.get(Calendar.HOUR_OF_DAY) - lastHours);

        MatchOperation match = Aggregation.match(Criteria.where("time").gte(calendar.getTime()));

        GroupOperation group = Aggregation.group()
                .sum("valid").as("validClickCount")
                .count().as("clickCount")
                .sum("price").as("consume");
        Aggregation aggregation = Aggregation.newAggregation(match, group);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Pipeline: {}", collectionName, aggregation);
        }
        AggregationResults<Count> output = mongoTemplate.aggregate(aggregation, collectionName, Count.class);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Finished aggregate last {} hours in {} ms", collectionName, lastHours, (System.currentTimeMillis() - start));
        }
        List<Count> results = output.getMappedResults();
        if (null == results) {
            return Count.empty();
        }
        Iterator<Count> iterator = results.iterator();
        return iterator.hasNext() ? iterator.next() : Count.empty();
    }

    private Count aggregate(String collectionName, String position, Long ticketId, int lastHours) {
        long start = System.currentTimeMillis();
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, calendar.get(Calendar.HOUR_OF_DAY) - lastHours);

        MatchOperation match = Aggregation.match(Criteria.where("time").gte(calendar.getTime()).and("position").is(position).and("ticketid").is(ticketId));

        GroupOperation group = Aggregation.group()
                .count().as("validClickCount");
        Aggregation aggregation = Aggregation.newAggregation(match, group);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Pipeline: {}", collectionName, aggregation);
        }
        AggregationResults<Count> output = mongoTemplate.aggregate(aggregation, collectionName, Count.class);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Finished aggregate last {} hours in {} ms", collectionName, lastHours, (System.currentTimeMillis() - start));
        }
        List<Count> results = output.getMappedResults();
        if (null == results) {
            return Count.empty();
        }
        Iterator<Count> iterator = results.iterator();
        return iterator.hasNext() ? iterator.next() : Count.empty();
    }

    private void createCollectionIfNecess(String collectionName) {
        String keyName = KeyBuilder.build("AD", "LOCK", collectionName);
        if (!lock.tryAcquire(keyName)) {
            return;
        }
        try {
            DB db = mongoTemplate.getDb();
            if (!db.collectionExists(collectionName)) {
                DBObject options = new BasicDBObject();
                options.put("capped", true);
                options.put("size", configuration.getOcpc().getMaxSize());
                options.put("max", configuration.getOcpc().getDocumentsMax());
                DBCollection collection = db.createCollection(collectionName, options);

                DBObject idxTime = new BasicDBObject();
                idxTime.put("time", -1);
                collection.createIndex(idxTime);
                DBObject idxBillid = new BasicDBObject();
                idxBillid.put("billid", -1);
                collection.createIndex(idxBillid);
            }
            COLLECTIONNAME_CACHE.add(collectionName);
        } finally {
            lock.release(keyName);
        }
    }

    private String createCollectionName(String position, BigInteger ticketId) {
        return StringUtils.join(new Object[] { COLLECTION_NAME_PREFIX, ticketId, position }, '_');
    }

    @Override
    public String getTopic() {
        return configuration.getTopic().getClick();
    }

    static class Count implements Serializable {

        private static final long serialVersionUID = -3784449261767131582L;
        private long consume;
        private long clickCount;
        private long validClickCount;

        static Count empty() {
            return new Count();
        }

        public long getConsume() {
            return consume;
        }

        public void setConsume(long consume) {
            this.consume = consume;
        }

        public long getClickCount() {
            return clickCount;
        }

        public void setClickCount(long clickCount) {
            this.clickCount = clickCount;
        }

        public long getValidClickCount() {
            return validClickCount;
        }

        public void setValidClickCount(long validClickCount) {
            this.validClickCount = validClickCount;
        }

        @Override
        public String toString() {
            return "Count{" +
                    "consume=" + consume +
                    ", clickCount=" + clickCount +
                    ", validClickCount=" + validClickCount +
                    '}';
        }
    }
}
