package com.bxm.adsprod.facade.ticket;

import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import org.apache.commons.lang.math.NumberUtils;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Map;

/**
 * <h3>广告券相关KEY创建者</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/12
 */
public final class
 TicketKeyGenerator {

    public static class Filter {
        private Filter() {}

        public static KeyGenerator getRegionDefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "REGION", "DEFINED");
        }

        public static KeyGenerator getRegionUndefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "REGION", "UNDEFINED");
        }

        public static KeyGenerator getPositionGroup(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_GROUP", ticketId);
        }

        public static KeyGenerator getPositionDefinedBlackList() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "BLACKLIST2");
        }

        public static KeyGenerator getPositionDefinedWhiteList() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "WHITELIST2");
        }

        public static KeyGenerator getPositionDefinedBlackConf() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "BLACKCONF2");
        }

        public static KeyGenerator getPositionDefinedWhiteConf() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "WHITECONF2");
        }

        public static KeyGenerator getPositionUndefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "UNDEFINED");
        }

        public static KeyGenerator getTimeline(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "FILTER", "TIMELINE", ticketId);
        }

        public static KeyGenerator getTicketGroup(BigInteger groupid) {
            return () -> KeyBuilder.build("AD", "FILTER", "TICKET_GROUP", groupid);
        }

        public static KeyGenerator getIpLibraryDefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "IP_DEFINED");
        }
        public static KeyGenerator getIpLibrary() {
            return () -> KeyBuilder.build("AD", "FILTER", "IP_LIB");
        }
    }

    public static class Advertiser {

        private Advertiser() {
        }

        public static KeyGenerator getBalance() {
            return () -> KeyBuilder.build("AD", "ADVERTISER", "BALANCE");
        }

        public static KeyGenerator getStationUserWeight() {
            return () -> KeyBuilder.build("STATION", "USER", "WEIGHT");
        }

        public static KeyGenerator getConsumeForStationUserWeight() {
            return () -> KeyBuilder.build("STATION", "USER", "WEIGHT", "CONSUME");
        }

        public static KeyGenerator getAdvertiserShop(BigInteger advertiserId){
            return () -> KeyBuilder.build("advertiser","adshop",advertiserId);
        }

    }

    public static class Media {
        private Media() {}

        public static KeyGenerator getPositions() {
            return () -> KeyBuilder.build("AD", "POSITION", "ALL");
        }

        /**
         * ADX广告位对应PositionId
         * @return
         */
        public static KeyGenerator getAdxPositionIds() {
            return () -> KeyBuilder.build("AD", "POSITION", "ADX", "ID");
        }

        public static KeyGenerator getPositionDomain(Map<String, Object> parameters) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_DOMAIN",
                    parameters.get(CachePushableFields.POSITION_ID),
                    parameters.get(CachePushableFields.DOMAIN_CODE));
        }

        public static KeyGenerator getPositionTag(Map<String, Object> parameters) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_TAG",
                    parameters.get(CachePushableFields.POSITION_ID),
                    parameters.get(CachePushableFields.TAG_CODE));

        }
        public static KeyGenerator getMediaTag(String positionId) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_TAG",positionId);
        }

        /**  @see    #getMediaTagHighestControllers()
         *
         */
        @Deprecated
        public static KeyGenerator getMediaTagHighestPositions() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_TAG_HIGHEST_POSITIONS");
        }

        /** 获取广告位标签从严配置项
         * @return
         */
        public static KeyGenerator getMediaTagHighestControllers() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_TAG_HIGHEST_CONTROLLERS");
        }

        /**
         *  媒体广告位测试
         */
        public static KeyGenerator getPositionTest(String positionId) {
            return () -> KeyBuilder.build("AD", "POSITION_TEST", positionId);
        }
        /**
         *  媒体广告位白名单
         */
        public static KeyGenerator getPositionWhiteTicketId(String positionId) {
            return () -> KeyBuilder.build("AD", "POSITION","WHITE","TICKETID", positionId);
        }
        /**
         *  开发者的用户信息
         */
        public static KeyGenerator getUserInfo(String appKey) {
            return () -> KeyBuilder.build("AD", "USER_MEDIA", "APPKEY", appKey);
        }
        /**
         * 激励视频appKey对应的广告位列表
         * @return
         */
        public static KeyGenerator getInspireAppKeys() {
            return () -> KeyBuilder.build("AD", "POSITION", "INSPIRE", "APPKEY");
        }

        /**
         * appKey对应的appSecret
         * @return
         */
        public static KeyGenerator getAppSecret() {
            return () -> KeyBuilder.build("AD", "MEDIA", "APPKEY");
        }
    }

    public static class User {
        private User() {}

        public static KeyGenerator getUser(String uid) {
            return () -> KeyBuilder.build("AD", "RECORD", "LAST", uid);
        }

        public static KeyGenerator getValidClickForTag(String uid) {
            return () -> KeyBuilder.build("AD", "RECORD", "VALIDCLICK", "TAG", uid);
        }

        public static KeyGenerator getValidClickForTagOfLast(String uid) {
            return () -> KeyBuilder.build("AD", "RECORD", "VALIDCLICK", "TAG_LAST", uid);
        }

        public static KeyGenerator getClosed(String uid) {
            return () -> KeyBuilder.build("AD", "CLOSED_TICKET", "UID", DateHelper.getDate(), uid);
        }

        public static KeyGenerator getClosedGroup(String uid) {
            return () -> KeyBuilder.build("AD", "CLOSED_TGROUP", "UID", DateHelper.getDate(), uid);
        }

        public static KeyGenerator getClosedTicket() {
            return () -> KeyBuilder.build("AD", "CLOSED_TICKET_ADX", DateHelper.getDate());
        }

        /**
         * 用户已安装的APP列表
         */
        public static KeyGenerator getInstalledAppList() {
            return () -> KeyBuilder.build("AD", "INSTALLED", "APP_LIST");
        }

        /**
         * 用户组/频次已到阈值的券列表
         * @param uid
         * @return
         */
        public static KeyGenerator getClosedTicketByUid(String uid) {
            return () -> KeyBuilder.build("AD", "USER_CLOSED_TICKET", DateHelper.getDate(), uid);
        }
    }

    public static class Statistics {

        private Statistics() {}

        public static KeyGenerator getBudgetOfDaily() {
            return getBudgetOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getBudgetOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "DAILY_BUDGET", date);
        }

        //ADX直投券-ADX的日预算
        public static KeyGenerator getAdxBudgetOfDaily() {
            return getAdxBudgetOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getAdxBudgetOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "DAILY_BUDGET", "ADX", date);
        }

        public static KeyGenerator getClickOfDaily() {
            return getClickOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getClickOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "CLICK",
                    date);
        }
        //有效点击
        public static KeyGenerator getValidClickOfDaily(BigInteger ticketId) {
            return getValidClickOfDaily(DateHelper.format("yyyyMMdd"),ticketId);
        }

        public static KeyGenerator getValidClickOfDaily(String date,BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "VALID_CLICK",date,ticketId);
        }

        public static KeyGenerator getViewOfDaily() {
            return getViewOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getViewOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "VIEW",
                    date);
        }

        public static KeyGenerator getPositionViewOfDaily(String position) {
            return getPositionViewOfDaily(DateHelper.format("yyyyMMdd"), position);
        }

        public static KeyGenerator getPositionViewOfDaily(String date, String position) {
            return () -> KeyBuilder.build("AD", "COUNTER", "VIEW",
                    date, position);
        }

        public static KeyGenerator getBudgetOfHourly() {
            return getBudgetOfHourly(NumberUtils.toInt(DateHelper.format("H"), -1));
        }

        public static KeyGenerator getBudgetOfHourly(int hour) {
            return () -> KeyBuilder.build("AD", "COUNTER", "HOURLY_BUDGET",
                    DateHelper.format("yyyyMMdd"),
                    hour);
        }

        public static KeyGenerator getViewOfHourly() {
            return getViewOfHourly(NumberUtils.toInt(DateHelper.format("H"), -1));
        }

        public static KeyGenerator getViewOfHourly(int hour) {
            return () -> KeyBuilder.build("AD", "COUNTER", "HOURLY_VIEW",
                    DateHelper.format("yyyyMMdd"),
                    hour);
        }

        public static KeyGenerator getClickOfDaily(BigInteger ticketId) {
            return getClickOfDaily(DateHelper.getDate(), ticketId);
        }

        public static KeyGenerator getClickOfDaily(String date, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_CLICK",
                    date,
                    ticketId);
        }

        public static KeyGenerator getViewOfDaily(BigInteger ticketId) {
            return getViewOfDaily(DateHelper.getDate(), ticketId);
        }

        public static KeyGenerator getViewOfDaily(String date, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_VIEW",
                    date,
                    ticketId);
        }

        public static KeyGenerator getClickOfWeekly(BigInteger ticketId) {
            return getClickOfWeekly(DateHelper.format("yyyyw"), ticketId);
        }

        public static KeyGenerator getClickOfWeekly(String dateweek, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_CLICK",
                    dateweek,
                    ticketId);
        }

        public static KeyGenerator getViewOfWeekly(BigInteger ticketId) {
            return getViewOfWeekly(DateHelper.format("yyyyw"), ticketId);
        }

        public static KeyGenerator getViewOfWeekly(String dateweek, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_VIEW",
                    dateweek,
                    ticketId);
        }

        public static KeyGenerator getBudgetOfPositionGroup(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "DAILY_BUDGET_P_G", DateHelper.format("yyyyMMdd"), ticketId);
        }

        /**
         * 获取广告组每天点击数
         *
         * @author 阿导
         * @time 2018/3/27
         * @CopyRight 杭州微财科技有限公司
         * @param groupId
         * @return com.bxm.warcar.cache.KeyGenerator
         */
        public static KeyGenerator getTicketGroupDailyClickKey(Long groupId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_CLICK_T_G", DateHelper.format("yyyyMMdd"), groupId);
        }
        /**
         * 获取广告组每天曝光数
         *
         * @author 阿导
         * @time 2018/3/27
         * @CopyRight 杭州微财科技有限公司
         * @param groupId
         * @return com.bxm.warcar.cache.KeyGenerator
         */
        public static KeyGenerator getTicketGroupDailyViewKey(Long groupId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_VIEW_T_G", DateHelper.format("yyyyMMdd"), groupId);
        }
        /**
         * 获取广告组每天点击数
         *
         * @author 阿导
         * @time 2018/3/27
         * @CopyRight 杭州微财科技有限公司
         * @param groupId
         * @return com.bxm.warcar.cache.KeyGenerator
         */
        public static KeyGenerator getTicketGroupWeekClickKey(Long groupId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_CLICK_T_G", DateHelper.format("yyyyw"), groupId);
        }
        /**
         * 获取广告组每天曝光数
         *
         * @author 阿导
         * @time 2018/3/27
         * @CopyRight 杭州微财科技有限公司
         * @param groupId
         * @return com.bxm.warcar.cache.KeyGenerator
         */
        public static KeyGenerator getTicketGroupWeekViewKey(Long groupId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_VIEW_T_G", DateHelper.format("yyyyw"), groupId);
        }

        //广告位下素材最近2个小时
        public static KeyGenerator getTicketAssets2Hour(String positionId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","2H", positionId);
        }
        //广告位下素材最近24个小时
        public static KeyGenerator getTicketAssets24Hour(String positionId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","24H", positionId);
        }
        //广告位下素材所有点击
        public static KeyGenerator getTicketAllAssetsClick(String positionId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","ALL","CLICKS", positionId);
        }
        //广告位下素材所有曝光
        public static KeyGenerator getTicketAllAssetsView(String positionId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","ALL","VIEWS", positionId);
        }

        //广告位下素材所有有效点击
        @Deprecated
        public static KeyGenerator getTicketAllAssetsValidClick(String positionId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","ALL","VALIDCLICKS", positionId);
        }

        /**
         * 广告位下素材累计有效点击
         *
         * @param positionId
         * @param assetsId
         * @return
         */
        public static KeyGenerator getTicketAllAssetsValidClick(String positionId, String assetsId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS", "ALL", "VALIDCLICKS", positionId, assetsId);
        }

        //广告位下素材当天点击
        public static KeyGenerator getTicketDailyAssetsClick() {
            return getTicketDailyAssetsClick(DateHelper.getDate());
        }
        //广告位下素材当天曝光
        public static KeyGenerator getTicketDailyAssetsView() {
            return getTicketDailyAssetsView(DateHelper.getDate());
        }
        //广告位下素材时间点点击
        public static KeyGenerator getTicketDailyAssetsClick(String daily) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","DAILY","CLICKS", daily);
        }
        //广告位下素材当时间点天曝光
        public static KeyGenerator getTicketDailyAssetsView(String daily) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","DAILY","VIEWS", daily);
        }


        //广告位下素材当天有效点击
        public static KeyGenerator getTicketDailyAssetsValidClick() {
            return getTicketDailyAssetsValidClick(DateHelper.getDate());
        }
        //广告位下素材时间点有效点击
        public static KeyGenerator getTicketDailyAssetsValidClick(String daily) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ASSETS","DAILY","VALIDCLICKS", daily);
        }



        //媒体广告位测试的曝光
        public static KeyGenerator getPositionIdTestViews(String positionid) {
            return () -> KeyBuilder.build("AD", "COUNTER", "POSITIONID_TEST","VIEWS", positionid);
        }
        //媒体广告位测试的点击
        public static KeyGenerator getPositionIdTestClick(String positionid) {
            return () -> KeyBuilder.build("AD", "COUNTER", "POSITIONID_TEST","CLICK", positionid);
        }
        //分段统计（每十分钟）券点击数
        public static KeyGenerator getAdxSectionClick(String hour, String section) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ADX_SECTION","CLICK", hour, section);
        }
        //分段统计（每十分钟）券曝光数
        public static KeyGenerator getAdxSectionShow(String hour, String section) {
            return () -> KeyBuilder.build("AD", "COUNTER", "ADX_SECTION","SHOW", hour, section);
        }
    }

    public static class Weight {

        @Deprecated
        public static KeyGenerator getWeight(BigInteger ticketId, String positionId) {
            return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET", ticketId);
        }

        public static KeyGenerator getWeight(String positionId) {
            return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET", positionId);
        }
    }

    public static class Scene {

        private Scene() {}

        public static KeyGenerator getScene(int scene) {
            return () -> KeyBuilder.build("AD", "TICKET", "SCENE", scene);
        }

        public static KeyGenerator getSceneViews(int scene, String uid) {
            return () -> KeyBuilder.build("AD", "COUNTER", "SCENE", scene, uid);
        }
    }

    public static class Ocpc {

        public static KeyGenerator getConsume(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "CONSUME", ticketId);
        }

        public static KeyGenerator getClick(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "CLICK", ticketId);
        }

        @Deprecated
        public static KeyGenerator getValidClick(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "VALIDCLICK", ticketId);
        }

        public static KeyGenerator getValidClick(BigInteger ticketId, String position) {
            return () -> KeyBuilder.build("AD", "OCPC", "VALIDCLICK", ticketId, position);
        }

        public static KeyGenerator getClosed(String position) {
            return () -> KeyBuilder.build("AD", "OCPC", "CLOSED", position);
        }

        public static KeyGenerator getFloor() {
            return () -> KeyBuilder.build("AD", "OCPC", "FLOOR");
        }

        public static KeyGenerator getOfferSheetLock(String position, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "OFFER_SHEET_LOCK", position, ticketId);
        }
        public static KeyGenerator getOfferSheet(String position, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "OFFER_SHEET", position, ticketId);
        }
    }

    public static class Temp {

        private Temp() {}

        public static KeyGenerator getTimelineOutOfBudget(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "TEMP", "TIMELINE", "OUT_OF_BUDGET", ticketId);
        }

        public static KeyGenerator getTickets(int os, int app, String regionCode, String position) {
            return () -> KeyBuilder.build("AD", "TEMP", "TICKET", os, app, regionCode, position);
        }
    }

    /**
     * 票据相关，这个不是广告券
     */
    public static class Ticket {

        public static final String F_SOURCE = "source";
        public static final String F_SHOW = "show";
        public static final String F_CLICK = "click";

        private Ticket() {}

        public static KeyGenerator getTicket(String ticket) {
            return () -> KeyBuilder.build("TICKET", ticket);
        }
    }

    public static class Threshold {

        private Threshold() {}

        public static KeyGenerator getLimitPriceConfig() {
            return () -> "AD:TICKET:THRESHOLD:LIMITPRICE_CONFIG";
        }

        public static KeyGenerator getLimitPriceEnabled() {
            return () -> "AD:TICKET:THRESHOLD:LIMITPRICE_ENABLED:" + DateHelper.getDate();
        }

        public static KeyGenerator getLimitPriceRemainder() {
            return () -> "AD:TICKET:THRESHOLD:LIMITPRICE_REMAINDER:" + DateHelper.getDate();
        }
    }

    public static KeyGenerator getUser(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "USER",
                parameters.get(CachePushableFields.IMEI));
    }

    public static KeyGenerator getTimes(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "FILTER", "TIMES",
                parameters.get(CachePushableFields.TICKET_ID));
    }

    public static KeyGenerator getTicketAssets() {
        return () -> KeyBuilder.build("AD", "TICKET", "ASSETS");
    }

    public static KeyGenerator getCouponsInfo(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "TICKET", "COUPONSINFO",
                parameters.get(CachePushableFields.TICKET_ID));
    }

    public static KeyGenerator getTicketAssetsCoupons() {
        return () -> KeyBuilder.build("AD", "TICKET", "ASSETS", "COUPONS");
    }

    public static KeyGenerator getTicketAdx(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "TICKET", "ADX",
                parameters.get(CachePushableFields.TICKET_ID));
    }

    public static KeyGenerator getTicketAssetsAdx() {
        return () -> KeyBuilder.build("AD", "TICKET", "ASSETS", "ADX");
    }

    public static KeyGenerator getTicketInspireVideo(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "TICKET", "INSPIREVIDEO",
                parameters.get(CachePushableFields.TICKET_ID));
    }

    public static KeyGenerator getTicketAssetsInspireVideo() {
        return () -> KeyBuilder.build("AD", "TICKET", "ASSETS", "INSPIREVIDEO");
    }

    public static KeyGenerator getAvailableTickets() {
        return () -> KeyBuilder.build("AD", "TICKET", "AVAILABLE");
    }

    public static KeyGenerator getAllTickets() {
        return () -> KeyBuilder.build("AD", "TICKET", "ALL");
    }

    public static KeyGenerator getTicketPrice(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketPriceOfTodayFirst(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE_TF",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }
    public static KeyGenerator getTicketPriceFloatPercent(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE_FP",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketMif(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_MIF",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketProfitForCpa(LocalDate date) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PROFIT_FOR_CPA", date.format(DateTimeFormatter.ofPattern("yyyyMMdd")));
    }

    /**
     * 广告位直投券 ,key 广告位 : value 券id_size list
     * @return
     */
    public static KeyGenerator getDirectPositionTicket(){
        return () -> KeyBuilder.build("AD","DIRECT", "POSITION"  , "TICKET");
    }

    /**
     * 直投券对应素材信息 ,key 券id_size : value 素材 list
     * @return
     */
//    public static KeyGenerator getDirectPositionTicketIdMaterial(){
//        return () -> KeyBuilder.build("AD","DIRECT", "POSITION"  , "TICKET" , "ID" ,"MATERIAL");
//    }

    /**
     * 广告位对应直投素材信息 ,key 广告位 : value 素材 list
     * @return
     */
    public static KeyGenerator getDirectPositionMaterial(){
        return () -> KeyBuilder.build("AD" ,"DIRECT" , "POSITION" ,"MATERIAL");
    }

    /**
     * 广告位对应CTR最后的直投素材信息 ,key 广告位 : value 素材 list
     * @return
     */
    public static KeyGenerator getDirectPositionMaterialCTR(String positionId){
        return () -> KeyBuilder.build("AD" ,"DIRECT" , "POSITION" ,"MATERIAL" ,"CTR",positionId);
    }

    /**
     * 直投券对应素材信息 ,key 素材ID : value 素材信息
     * @return
     */
    public static KeyGenerator getDirectTicketMaterial(){
        return () -> KeyBuilder.build("AD" ,"DIRECT"  , "POSITION" ,"TICKET","MATERIAL");
    }

    /**
     * 直投券对应广告ID
     * AD:DIRECT:TICKET:ID:POSITION
     * @return
     */
    public  static KeyGenerator getDirectTicketIDPosition(){
        return () -> KeyBuilder.build("AD"  ,"DIRECT" ,"TICKET","ID" , "POSITION");
    }

    /**
     * 历史曝光
     * @param positionId
     * @return
     */
    public  static KeyGenerator getDirectMaterialView(String positionId){
        return () -> KeyBuilder.build("AD","DIRECT","TICKET","MATERIAL","VIEW",positionId);
    }

    /**
     * 历史点击
     * @param positionId
     * @return
     */
    public  static KeyGenerator getDirectMaterialClick(String positionId){
        return () -> KeyBuilder.build("AD","DIRECT","TICKET","MATERIAL","CLICK",positionId);
    }

    /**
     * @return 落地页需要重设的广告位集合
     */
    public static KeyGenerator landingResetPosition() {
        return () -> KeyBuilder.build("AD", "TICKET", "LANDING", "RESET", "POSITIONS");
    }

    /**
     * @return 落地页需要重设的广告集合
     */
    public static KeyGenerator landingResetTicket() {
        return () -> KeyBuilder.build("AD", "TICKET", "LANDING", "RESET", "TICKETS");
    }


    /**
     * 自建站相关
     */
    public static class DIYSite {

        /**
         * 自建站页面信息(hash)
         * AD:SITE:DIY:PAGE
         * @return
         */
        public static KeyGenerator getPageInfo() {
            return () -> KeyBuilder.build("AD", "SITE", "DIY", "PAGE");
        }

    }

    /**
     *拼多多的券
     */
    public static KeyGenerator getPddTickets(){
        return () -> KeyBuilder.build("AD" ,"PDD","TICKETS");
    }

    /**
     * 获取广告位代号的key
     */
    public static KeyGenerator getPositionCodeKey() {
        return () -> KeyBuilder.build("AD", "POSITION", "CODE");
    }
    /**
     * 广告位行业开关的key
     *
     * @return
     */
    public static KeyGenerator getKeyGeneratorByPositionDomainiCode() {
        return () -> KeyBuilder.build("AD", "POSITION", "DOMAINCODE");
    }
    /**
     * 广告所有的属性标签
     *
     * @return
     */
    public static KeyGenerator getAttributeTag() {
        return () -> KeyBuilder.build("AD", "ATTRIBUTE", "TAG");
    }

    /**
     * 广告票券码
     */
    public static class Coupon {

        /**
         * AD:TICKET:{ticket_id}:COUPONS:TYPE:{type}
         * @param ticket_id 广告票券ID
         * @param type -1已锁定(临时值)，0未使用，1待使用，2已使用
         * @return
         */
        public static KeyGenerator getCouponInfo(BigInteger ticket_id, Short type) {
            return () -> KeyBuilder.build("AD", "TICKET", ticket_id, "COUPONS", "TYPE", type);
        }

    }

    /**
     * ADX广告创意
     */
    public static class Adx {

        /**
         * 数据中心-2小时内有素材CTR信息的广告位列表
         */
        public static KeyGenerator getPositionListForDC() {
            return () -> KeyBuilder.build("ADX", "POSITION", "MATERIAL", "2H");
        }

        /**
         * 数据中心-2小时内广告位下的素材CTR信息
         */
        public static KeyGenerator getCreativeCtrForDC(String positionId) {
            return () -> KeyBuilder.build("ADX", "POSITION", "MATERIAL", "2H", positionId);
        }

        /**
         * 广告位常规素材
         */
        public static KeyGenerator getPositionNormal() {
            return () -> KeyBuilder.build("AD", "ADX", "POSITION", "CREATIVE", "NORMAL");
        }

        /**
         * ADX券包
         */
        public static KeyGenerator getTicketPack() {
            return () -> KeyBuilder.build("AD", "ADX", "TICKET_PACK");
        }

        /**
         * 缓存个推子弹夹token
         * @return
         */
        public static KeyGenerator getGeTuiToken() {
            return () -> KeyBuilder.build("AD", "ADX", "GETUI", "CLIP", "TOKEN");
        }
    }
    public static class Position {

        @Deprecated
        public static KeyGenerator getClosed(String positionId) {
            return () -> KeyBuilder.build("AD", "CLOSED_TICKET", "POSITION", DateHelper.getDate(), positionId);
        }

        public static KeyGenerator getClosedTicket(String groupId) {
            return () -> KeyBuilder.build("AD", "CLOSED_TICKET", "GROUP", DateHelper.getDate(), groupId);
        }
    }

    /**
     * 人群包
     */
    public static class CrowdPack {

        /**
         * 第三方人群包
         * Map<CROWDPACKID, CROWDPACK>
         */
        public static KeyGenerator getThirdCrowdPack() {
            return () -> KeyBuilder.build("AD", "CROWD_PACK", "THIRD");
        }

        /**
         * 第三方人群包
         * Map<TicketId, Set<CROWDPACKID-projectId>>
         */
        public static KeyGenerator getTicketCrowdPack() {
            return () -> KeyBuilder.build("AD", "CROWD_PACK", "THIRD", "TICKET");
        }
    }


    /**
     * fm出券算法相关
     */
    public static  class TicketFm {

        /**  用户全局发券
         * @param uid
         * @return
         */
        public static KeyGenerator getSeqLatelyUser(String uid){
            return () ->KeyBuilder.build("ADA","SEQ","LATELY","USER",uid);
        }


        public static KeyGenerator getSeqCount(Date date, long id){
            return () ->KeyBuilder.build(DateHelper.convertDateToString(date,DateHelper.PATTERN_STR8),id);
        }

        public static KeyGenerator getSeqCountNotTicketId(Date date){
            return () ->KeyBuilder.build(DateHelper.convertDateToString(date,DateHelper.PATTERN_STR8));
        }


        public static KeyGenerator getSeqCount(String dateStr, long id){
            return () ->KeyBuilder.build(dateStr,id);
        }

        /** spm 相关
         * @return
         */
        public static KeyGenerator getSpm(){
            return () ->KeyBuilder.build("ADA","SPM","USERS",DateHelper.getDate());
        }


        /** 用户当日参与数据
         * @param uid
         * @return
         */
        public static KeyGenerator getCurrentUser(String uid){
            return () ->KeyBuilder.build("ADA","SEQ","REPEAT","DATE",DateHelper.getDate(),"USER",uid);
        }


        /** 活动spm 相关
         * @param activityId
         * @return
         */
        public static KeyGenerator getCurrentActivity(long activityId){
            return () ->KeyBuilder.build("ADA","SEQ","ACTIVITY",activityId,"BYSPM");
        }


        /** 用户当日发券次数
         * @return
         */
        public static KeyGenerator getTicketCountCurrentDay(){
            return () ->KeyBuilder.build("ADA","SEQ");
        }


        /**  用户当日第几次重复发该券
         * @param ticketId
         * @return
         */
        public static KeyGenerator getTicketCountRepeat(long ticketId){
            return () ->KeyBuilder.build("ADA","REPEAT","ACTIVITY",ticketId);
        }


        /** 用户当日该活动订单次序
         * @param activityId
         * @return
         */
        public static KeyGenerator getActivityCount(long activityId){
            return () ->KeyBuilder.build("ADA","SEQ","ACTIVITY",activityId);
        }


        /** 用户当天上一单的活动或者券id,与券id的三级标签
         * @return
         */
        public static KeyGenerator getPrevTicketInfo(){
            return () ->KeyBuilder.build("ADA","RECORD","USERS","DATE",DateHelper.getDate());
        }


        /**  用户近期参与数据
         * @param uid
         * @return
         */
        public static KeyGenerator getLaterUser(String uid){
            return () ->KeyBuilder.build("ADA","RECORD","LATELY","USER",uid);
        }


        /**用户全局上一单发券曝光时间
         * @return
         */
        public static KeyGenerator getPrevShowDate(){
            return () ->KeyBuilder.build("ADA","RECORD","LATELY");
        }

        /** 用户全局该活动上一单发券曝光时间
         * @param activityId
         * @return
         */
        public static KeyGenerator getPrevActivityShowDate(long  activityId){
            return () ->KeyBuilder.build("ADA","RECORD","LATELY","ACTIVITY",activityId);
        }

        /** 全局前一单计费金额
         * @return
         */
        public static KeyGenerator getPrevCost(){
            return () ->KeyBuilder.build("ADA","COST","LATELY");
        }


        /** 全局前一单活动计费金额
         * @param activityId
         * @return
         */
        public static KeyGenerator getPrevCost(long  activityId){
            return () ->KeyBuilder.build("ADA","COST","LATELY","ACTIVITY",activityId);
        }



        /** 用户每天每张券出券次数
         * @param uid
         * @return
         */
        public static KeyGenerator getUserEveryTicketCount(String uid){
            return () ->KeyBuilder.build("ADA","RECORD","UDAILY_VIEW",DateHelper.getDate(),uid);
        }

        /** fm 日志存储key (hash)
         * @param bxmId
         * @return
         */
        public static KeyGenerator getFmLogByBxmId(String bxmId) {
            return () -> KeyBuilder.build("ADA", "TICKET", "FM", bxmId);
        }


        /** fm日志存储 最终出券id field
         * @return
         */
        public static KeyGenerator getFmLogFinalTicketId() {
            return () -> KeyBuilder.build("BEST","TICKET");
        }


        /** fm日志存储 出券列表 field
         * @return
         */
        public static KeyGenerator getFmLogTicketList() {
            return () -> KeyBuilder.build("SORTLIST");
        }

    }

    /**
     * 标签相关
     */
      public static  class Tags {
          public static KeyGenerator getTags(){
              return () ->KeyBuilder.build("ADA","TAGS");
          }
      }

      public static class Activity{

          /**活动基本信息缓存**/
          public static KeyGenerator getActBasicMsg(Long activityId) {
              return () -> KeyBuilder.build("AD", "ACTIVITY", "BASIC", "MSG", activityId);
          }
      }

    /**
     * 对外接口相关
     */
    public static class API {

        /**
         * 失败次数
         * @return
         */
        public static KeyGenerator getRequestFail(){
            return () ->KeyBuilder.build("API","DOT","REQUEST","FAIL");
        }

        /**
         * 请求信息
         * @param req_id 请求ID
         * @return
         */
        public static KeyGenerator getRequestInfo(String req_id){
            return () ->KeyBuilder.build("API","DOT","REQUEST", req_id);
        }

    }
}
