package com.bxm.adsprod.timer.jobs;

import com.alibaba.dubbo.config.annotation.Reference;
import com.bxm.adsprod.facade.advertiser.AdvertiserService;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.TicketService;
import com.bxm.adsprod.facade.ticket.TicketStatisticsService;
import com.bxm.adsprod.model.so.rules.TicketTimelineRuleSo;
import com.bxm.adsprod.timer.Job;
import com.bxm.warcar.cache.Fetcher;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * @author allen
 * @since 1.0.0
 */
@Component
public class TicketStatusRefreshJob implements Job {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketStatusRefreshJob.class);

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Reference(version = "1.0.0")
    private TicketService ticketService;
    @Reference(version = "1.0.0")
    private AdvertiserService advertiserService;
    @Reference(version = "1.0.0")
    private TicketStatisticsService ticketStatisticsService;

    @Override
    @Scheduled(cron = "0 0 * * * ?")
    public void execute() {
        long start = System.currentTimeMillis();

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Starting refresh status...");
        }

        List<Ticket> tickets = ticketService.getAllTickets();
        if (CollectionUtils.isNotEmpty(tickets)) {

            List<Ticket> refreshIfNeeded = Lists.newArrayList();

            for (Ticket ticket : tickets) {
                BigInteger ticketId = ticket.getId();
                Byte status = ticket.getStatus();
                Integer pauseReason = defaultIfEmpty(ticket.getPauseReason(), Ticket.PAUSE_REASON_NONE);
                Date startDate = ticket.getValidStartDate();
                Date endDate = ticket.getValidEndDate();
                if (status == Ticket.STATUS_DELETED) {
                    continue;
                }

                // 结束了
                boolean isOpenOrPause = status == Ticket.STATUS_OPEN || status == Ticket.STATUS_PAUSE;
                if (isOpenOrPause && pauseReason != Ticket.PAUSE_REASON_ENDED && isAfter(endDate)) {
                    ticket.setStatus(Ticket.STATUS_PAUSE);
                    ticket.setPauseReason(Ticket.PAUSE_REASON_ENDED);
                    refreshIfNeeded.add(ticket);
                    continue;
                }

                // 从未开始到开始
                if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_NO_START
                        && isEqual(startDate)) {
                    // 还要判断余额是否足够
                    setOpenIfEnought(refreshIfNeeded, ticket);
                }

                // 日预算不足 在0点执行的时候需要重新开启
                if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_OUT_OF_DAILYBUDGET
                        && isMidnight()) {
                    setOpenIfEnought(refreshIfNeeded, ticket);
                }

                // 时段预算开关
                TicketTimelineRuleSo timelineConf = getTimelineConf(ticketId);
                if (null != timelineConf) {
                    boolean closeIfNeeded = isCloseIfNeeded(ticketId, timelineConf.getEntries());
                    if (status == Ticket.STATUS_OPEN && closeIfNeeded) {
                        // 时段预算不足
                        ticket.setStatus(Ticket.STATUS_PAUSE);
                        ticket.setPauseReason(Ticket.PAUSE_REASON_OUT_OF_TIMEBUDGET);
                        refreshIfNeeded.add(ticket);
                    } else if (status == Ticket.STATUS_PAUSE && pauseReason == Ticket.PAUSE_REASON_OUT_OF_TIMEBUDGET
                            && !closeIfNeeded) {
                        // 时段预算不足需要重新打开
                        ticket.setStatus(Ticket.STATUS_OPEN);
                        ticket.setPauseReason(Ticket.PAUSE_REASON_NONE);
                        refreshIfNeeded.add(ticket);
                    }
                }
            }

            for (Ticket ticket : refreshIfNeeded) {
                if (!ticketService.updateTicketStatus(ticket.getId(), ticket.getStatus(), ticket.getPauseReason())) {
                    if (LOGGER.isWarnEnabled()) {
                        LOGGER.warn("[{}] Updating status to {}-{} return fail", ticket.getId(), ticket.getStatus(), ticket.getPauseReason());
                    }
                } else {
                    if (LOGGER.isInfoEnabled()) {
                        LOGGER.info("[{}] Updating status to {}-{} success", ticket.getId(), ticket.getStatus(), ticket.getPauseReason());
                    }
                }
            }
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Finished in {} ms", (System.currentTimeMillis() - start));
        }
    }

    private int defaultIfEmpty(Integer v, int defaultValue) {
        return null == v ? defaultValue : v;
    }

    private void setOpenIfEnought(List<Ticket> refreshIfNeeded, Ticket ticket) {
        boolean isCPA = !ticket.isCPC();
        if (isEnought(ticket.getAdvertiser()) || isCPA) {
            ticket.setStatus(Ticket.STATUS_OPEN);
            ticket.setPauseReason(Ticket.PAUSE_REASON_NONE);
            refreshIfNeeded.add(ticket);
        } else {
            ticket.setStatus(Ticket.STATUS_PAUSE);
            ticket.setPauseReason(Ticket.PAUSE_REASON_BALANCE_NOT_ENOUGHT);
            refreshIfNeeded.add(ticket);
        }
    }

    private boolean isEnought(BigInteger advertiser) {
        Long balance = advertiserService.getAdvertiserBalance(advertiser);
        return (null != balance && balance > 0);
    }

    private boolean isCloseIfNeeded(BigInteger ticketId, List<TicketTimelineRuleSo.Entry> entries) {
        int currentHour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY);
        for (TicketTimelineRuleSo.Entry entry : entries) {
            int startHour = entry.getStartHour();
            int endHour = entry.getEndHour();
            boolean isNow = currentHour >= startHour && currentHour < endHour;
            if (isNow) {
                long currentBudget = ticketStatisticsService.getBudgetOfTimeline(ticketId, startHour, endHour);
                return (currentBudget >= entry.getLimit());
            }
        }
        return true;
    }

    private TicketTimelineRuleSo getTimelineConf(BigInteger ticketId) {
        return fetcher.fetch(TicketKeyGenerator.Filter.getTimeline(ticketId), null, TicketTimelineRuleSo.class);
    }

    private static boolean isMidnight() {
        return LocalDateTime.now().getHour() == 0;
    }

    private static boolean isAfter(Date date) {
        return LocalDate.now().isAfter(toLocalDate(date));
    }

    private static boolean isEqual(Date date) {
        return LocalDate.now().isEqual(toLocalDate(date));
    }

    private static LocalDate toLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }
}
