package com.bxm.adx.common.sell.position;

import java.util.Map;
import java.util.Objects;

import org.springframework.stereotype.Component;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;

/**
 *
 * For cache pushable
 *
 * <pre>
 * name = "POSITION"
 * parameters = {positionId=$positionId}
 * byte[] = Position object to json bytes
 * </pre>
 * @author allen
 * @since 2019-12-18
 */
@CachePush("POSITION")
@Component
public class PositionDaoImpl implements PositionDao, Pushable {

    private final Fetcher fetcher;
    private final Updater updater;

    private static final String APPID = "APPID";
    private static final String PID = "PID";

    public PositionDaoImpl(Fetcher fetcher, Updater updater) {
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public Position getByAppPositionId(String appPositionId) {
        return fetcher.fetch(getKeyGenerator(APPID, appPositionId), Position.class);
    }

    @Override
    public Position getByPositionId(String positionId) {
        return fetcher.fetch(getKeyGenerator(PID, positionId), Position.class);
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        Object o = parameters.get(CachePushableFields.POSITION_ID);
        if (Objects.isNull(o)) {
            return;
        }
        String positionId = Objects.toString(o);
        Position position = JsonHelper.convert(data, Position.class);

        updater.remove(getKeyGenerator(APPID, position.getAppPositionId()));
        updater.remove(getKeyGenerator(PID, positionId));

        if (position.isEnabled()) {
            updater.update(getKeyGenerator(APPID, position.getAppPositionId()), position);
            updater.update(getKeyGenerator(PID, positionId), position);
        }
    }

    private KeyGenerator getKeyGenerator(String name, String value) {
        return () -> KeyBuilder.build("ADX", "POSITION", name, value);
    }
}
