package com.bxm.adx.common.buy.buyers;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.DispatcherService;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.plugin.PluginHolder;
import com.bxm.adx.common.sell.position.Position;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @author allen
 * @since 2019-12-16
 */
@Primary
@Configuration
public class DispatcherPluginBuyersImpl extends AbstractPluginBuyersImpl {

    private DispatcherService dispatcherService;
    private final AdxProperties properties;

    protected DispatcherPluginBuyersImpl(PluginHolder pluginHolder, AdxProperties properties) {
        super(pluginHolder);
        this.properties = properties;
    }

    @Override
    public List<List<Buyer>> findAsPriority(Position position) {
        String positionId = position.getPositionId();
        Map<Integer, Collection<Dsp>> dsps = dispatcherService.getPriority(positionId);

        //此处因为系统暂时只支持5级优先级分配，所以设置size为5就够了，实际如果想兼容不限级数的优先级的话需要考虑list的容量动态扩容的问题，
        //list作为接口是没有这个方法的，arraylist有，但是此处list对象不能直接调用arraylist类的方法
        List<List<Buyer>> list = emptyListWithCapacity(properties.getMaxCapacityOfBuyers());

        Map<String, Buyer> buyers = getAllBuyers();
        for (Map.Entry<Integer, Collection<Dsp>> entry : dsps.entrySet()) {
            // 优先级，有可能从2开始
            Integer priority = entry.getKey();
            Collection<Dsp> value = entry.getValue();

            int index = priority - 1;
            List<Buyer> buyerList = CollectionUtils.isEmpty(list) ? null : list.size() > index ? list.get(index) : null;

            // 如果当前优先级队列不存在，则构建一个队列
            if (null == buyerList) {
                buyerList = Lists.newArrayList();
                list.add(index, buyerList);
            }

            // 将Buyer实例添加到队列中
            for (Dsp dsp : value) {
                String dspCode = dsp.getDspCode();
                Buyer buyer = buyers.get(dspCode);
                if (null != buyer) {
                    buyerList.add(buyer);
                }
            }
        }

        // 删除没有空的队列
        list.removeIf(List::isEmpty);

        return list;
    }

    private List<List<Buyer>> emptyListWithCapacity(int size) {
        List<List<Buyer>> list = Lists.newArrayList();
        for (int i = 0; i < size; i++) {
            list.add(Lists.newArrayList());
        }
        return list;
    }

    @Autowired
    public void setDispatcherService(DispatcherService dispatcherService) {
        this.dispatcherService = dispatcherService;
    }
}
