package com.bxm.adx.common.sell.position.dsp;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.common.sell.creatives.MediaEntranceCreativesBuider;
import com.bxm.adx.common.utils.DateUtils;
import com.bxm.mccms.facade.model.pushable.SceneSettingCacheVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 *
 * For cache pushable
 *
 * <pre>
 * name = "POSITION"
 * parameters = {positionId=$positionId}
 * byte[] = SceneSetting object to json bytes
 * </pre>
 * @author zhengwangeng
 * @since 2020-5-28
 */
@CachePush("SCENE_SETTING")
@Component
@Slf4j
public class SceneSettingDaoImpl implements SceneSettingDao, Pushable {

    private final Fetcher fetcher;
    private final Updater updater;

    public SceneSettingDaoImpl(Fetcher fetcher, Updater updater) {
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public SceneSettingCacheVO getByPositionId(String positionId) {
        return fetcher.fetch(getKeyGenerator(positionId), SceneSettingCacheVO.class);
    }

    @Override
    public Map<Integer, String> getActivityIdByPosId(String positionId) {
        LocalDate now = LocalDate.now();
        String date = now.format(DateTimeFormatter.BASIC_ISO_DATE);
        String testActId = fetcher.hfetch(getRelationTestKeyGenerator(positionId), date, String.class);
        String normalActId = fetcher.hfetch(getRelationKeyGenerator(positionId), date, String.class);
        Map<Integer, String> result = new HashMap<>();
        result.put(1, normalActId);
        result.put(2, testActId);
        return result;
    }

//    @Override
//    public SceneActivity getSceneActivity(String activityId) {
//        return fetcher.fetch(getActivityKeyGenerator(activityId), SceneActivity.class);
//    }

    @Override
    public boolean isParticipate(String positionId, String uid, String activityId) {
        String aid = fetcher.hfetch(getParticipateKeyGenerator(positionId), uid, String.class);
        if (StringUtils.isEmpty(aid)) {
            return false;
        } else {
            if (aid.equals(activityId)) {
                return true;
            } else {
                return false;
            }
        }
    }

    @Override
    public void participate(String positionId, String uid, String activityId) {
        updater.hupdate(getParticipateKeyGenerator(positionId), uid, activityId);
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        if (log.isWarnEnabled()) {
            for (Map.Entry<String, Object> entry : parameters.entrySet()) {
                log.warn("k = {} , v = {}", entry.getKey(), entry.getValue());
            }
        }
        Object o = parameters.get(CachePushableFields.POSITION_ID);
        if (Objects.isNull(o)) {
            return;
        }
        String positionId = Objects.toString(o);
        SceneSettingCacheVO sceneSetting = JsonHelper.convert(data, SceneSettingCacheVO.class);
        if (log.isWarnEnabled()) {
            log.warn("sceneSetting = {}", sceneSetting);
        }
        updater.remove(getKeyGenerator(positionId));

        updater.update(getKeyGenerator(sceneSetting.getPositionId()), sceneSetting);

        final List<Long> creativeIds = sceneSetting.getMediaEntranceCreatives().stream().
                filter(SceneSettingCacheVO.MediaEntranceCreative::isOpen)
                .map(SceneSettingCacheVO.MediaEntranceCreative::getCreativeId)
                .collect(Collectors.toList());
        updater.hupdateWithSelector(MediaEntranceCreativesBuider.positionAllCreatives(),positionId, JSON.toJSONString(creativeIds),2);

        handleActivity(positionId, sceneSetting.getSceneActivities());
    }

    /**
     * 记录用户上一次参与的活动
     * @param value
     * @return
     */
    private KeyGenerator getParticipateKeyGenerator(String value) {
        return () -> KeyBuilder.build("ADX", "POSITION", "SCENE", "PARTICIPATE", value);
    }

    /**
     * 记录 场景对象
     * @param value
     * @return
     */
    private KeyGenerator getKeyGenerator(String value) {
        return () -> KeyBuilder.build("ADX", "POSITION", "SCENE", "DSP", value);
    }

    /**
     * 广告位和对应正式活动的关系
     * 数据结构：hash （k,v）->(date, activityId)
     * @param positionId
     * @return
     */
    private KeyGenerator getRelationKeyGenerator(String positionId) {
        return () -> KeyBuilder.build("ADX", "POSITION", "SCENE", "ACTIVITY", "RELATION", positionId);
    }

    /**
     * 广告位对应测试活动的关系
     * 数据结构：hash （k,v）->(date, activityId)
     *
     * @param positionId
     * @return
     */
    private KeyGenerator getRelationTestKeyGenerator(String positionId) {
        return () -> KeyBuilder.build("ADX", "POSITION", "SCENE", "ACTIVITY", "RELATION", "TEST", positionId);
    }

//    private KeyGenerator getActivityKeyGenerator(String activityId) {
//        return () -> KeyBuilder.build("ADX", "POSITION", "SCENE", "ACTIVITY", activityId);
//    }

    private void handleActivity(String pid, List<SceneSettingCacheVO.SceneActivity> sceneActivities) {
        updater.remove(getRelationKeyGenerator(pid));
        updater.remove(getRelationTestKeyGenerator(pid));
        if (CollectionUtils.isEmpty(sceneActivities)) {
            return;
        }
        for (SceneSettingCacheVO.SceneActivity sa : sceneActivities) {
            if (sa.getOpened() == 0) {
                continue;
            }
            String aid = sa.getActivityId();
//            updater.update(getActivityKeyGenerator(aid), sa);

            LocalDate start = DateUtils.convertDate(sa.getStartDate());
            LocalDate end = DateUtils.convertDate(sa.getEndDate());
            Integer type = sa.getType();
            while (start.compareTo(end) <= 0) {
                String date = start.format(DateTimeFormatter.BASIC_ISO_DATE);
                if (1 == type) {
                    updater.hupdate(getRelationKeyGenerator(pid), date, aid);
                } else {
                    updater.hupdate(getRelationTestKeyGenerator(pid), date, aid);
                }
                start = start.plusDays(1);
            }
        }
    }
}
