package com.bxm.adx.common;

import lombok.Data;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Objects;

/**
 * @author fgf
 * @date 2022-06-02
 * @since 1.0
 */
@Data
public class OpenLogProperties {

    /**
     * http / https
     */
    private String scheme = "https";

    /**
     * internal http / https
     */
    private String internalScheme = "https";

    /**
     * hostname
     */
    private String host = "openlog.bianxianmao.com";

    /**
     * internal hostname
     */
    private String internalHost = "openlog-inner.bianxianmao.com";

    /**
     * port, default: -1
     */
    private int port = -1;

    /**
     * internal port, default: -1
     */
    private int internalPort = -1;

    private String reqDomain;

    /**
     * OpenLog 日志订阅者，事件模式的 消息队列消费者组 ID
     */
    private String consumerGroupForEventBus = "ADX";

    public String getReqDomain() {
        if (StringUtils.isBlank(reqDomain)) {
            this.reqDomain = UriComponentsBuilder.newInstance().scheme(getScheme()).host(getHost()).port(getPort()).build().toString();
        }
        return this.reqDomain;
    }

    /**
     * query string
     */
    private String query = "/adx/s.gif" +
            "?mt=" + OpenlogConstants.Macros.MT +
            "&ver=" + OpenlogConstants.Macros.VER +
            "&bxmid=" + OpenlogConstants.Macros.BXMID +
            "&time=" + OpenlogConstants.Macros.TIMESTAMP +
            "&bidid=" + OpenlogConstants.Macros.BIDID +
            "&tagid=" + OpenlogConstants.Macros.TAGID +
            "&price=" + OpenlogConstants.Macros.PRICE +
            "&configid=" + OpenlogConstants.Macros.CONFIGID +
            "&dspid=" + OpenlogConstants.Macros.DSPID +
            "&appid=" + OpenlogConstants.Macros.APPID +
            "&dappid=" + OpenlogConstants.Macros.DAPPID +
            "&dtagid=" + OpenlogConstants.Macros.DTAGID +
            "&dpr=" + OpenlogConstants.Macros.DPRICE +
            "&medid=" + OpenlogConstants.Macros.MEDIA_ID +
            "&chgtypm=" + OpenlogConstants.Macros.CHARGE_TYPE_MEDIA +
            "&chgtypd=" + OpenlogConstants.Macros.CHARGE_TYPE_DSP +
            "&device=" + OpenlogConstants.Macros.DEVICE
            ;

    /**
     * 获取盘古埋点完整的URL地址
     * @param mt 将替换宏参 __MT__ 的值
     * @return URL
     */
    public String create(int mt) {
        String queryString = this.query.replaceAll(OpenlogConstants.Macros.MT, Objects.toString(mt));
        return UriComponentsBuilder.fromUriString(queryString)
                .scheme(getScheme())
                .host(getHost())
                .port(getPort())
                .build()
                .toString();
    }

    /**
     * 获取盘古埋点完整的URL 内网地址
     * @param mt 将替换宏参 __MT__ 的值
     * @return URL
     */
    public String createInternal(int mt) {
        String queryString = this.query.replaceAll(OpenlogConstants.Macros.MT, Objects.toString(mt));
        return UriComponentsBuilder.fromUriString(queryString)
                .scheme(getInternalScheme())
                .host(getInternalHost())
                .port(getInternalPort())
                .build()
                .toString();
    }

    /**
     * 获取盘古埋点完整的URL地址
     * @param mt 将替换宏参 __MT__ 的值
     * @param removeParams 需要删除的参数名，比如传入：time，那么返回的URL里不会包含这个参数。
     * @return URL
     */
    public String create(int mt, String...removeParams) {
        return this.create(mt, null, removeParams);
    }

    /**
     * 获取盘古埋点完整的URL地址
     * @param internal 是否走内网
     * @param mt 将替换宏参 __MT__ 的值
     * @param removeParams 需要删除的参数名，比如传入：time，那么返回的URL里不会包含这个参数。
     * @return URL
     */
    public String create(boolean internal, int mt, String...removeParams) {
        return this.create(internal, mt, null, removeParams);
    }

    /**
     * 获取盘古埋点完整的URL地址
     * @param mt 将替换宏参 __MT__ 的值
     * @param customizeParams 自定义参数。可新增或替换原有URL上的参数及参数值。
     * @param removeParams 需要删除的参数名，比如传入：time，那么返回的URL里不会包含这个参数。
     * @return URL
     * @see #create(boolean, int, MultiValueMap, String...)
     */
    public String create(int mt, MultiValueMap<String, String> customizeParams, String...removeParams) {
        return create(false, mt, customizeParams, removeParams);
    }

    /**
     * 获取盘古埋点完整的URL地址
     * @param internal 是否走内网
     * @param mt 将替换宏参 __MT__ 的值
     * @param customizeParams 自定义参数。可新增或替换原有URL上的参数及参数值。
     * @param removeParams 需要删除的参数名，比如传入：time，那么返回的URL里不会包含这个参数。
     * @return URL
     */
    public String create(boolean internal, int mt, MultiValueMap<String, String> customizeParams, String...removeParams) {
        String url = internal ? createInternal(mt) : create(mt);
        if (MapUtils.isEmpty(customizeParams) && ArrayUtils.isEmpty(removeParams)) {
            return url;
        }
        UriComponentsBuilder builder = UriComponentsBuilder.fromUriString(url);
        UriComponents uriComponents = builder.build();

        MultiValueMap<String, String> params = uriComponents.getQueryParams();
        MultiValueMap<String, String> clone = new LinkedMultiValueMap<>(params);
        if (Objects.nonNull(customizeParams)) {
            clone.addAll(customizeParams);
        }
        for (String exclude : removeParams) {
            clone.remove(exclude);
        }
        builder.replaceQueryParams(clone);

        return builder.build().toString();
    }
}
