package com.bxm.adx.common.buy.cache;

import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * DSP-Response 缓存
 *
 * @author fgf
 * @date 2022/7/6
 **/
@Service
@Slf4j
public class BuyerResponseCache implements ResponseCache {
    private final Updater updater;
    private final Fetcher fetcher;

    public BuyerResponseCache(Updater updater, Fetcher fetcher) {
        this.updater = updater;
        this.fetcher = fetcher;
    }

    @Override
    public void saveResponse(BidRequest bidRequest, BidResponse bidResponse, Buyer buyer) {
        KeyGenerator key = buildKey(bidRequest, buyer);
        Dsp dsp = buyer.getDsp();
        if (Objects.isNull(dsp.getCacheTime()) || dsp.getCacheTime() == 0) {
            if (log.isDebugEnabled()) {
                log.debug("dsp don't support cache");
            }
        }
        if (Objects.isNull(key)) {
            if (log.isDebugEnabled()) {
                log.debug("uid is null , fail to save response");
            }
            return;
        }
        if (Objects.nonNull(bidResponse)) {
            try {
                String response = new ObjectMapper().writeValueAsString(bidResponse);
                if (!dsp.isForbidden() && Objects.nonNull(dsp.getCacheTime())) {
                    updater.update(key, response, dsp.getCacheTime().intValue());
                }
            } catch (JsonProcessingException e) {
                if (log.isErrorEnabled()) {
                    log.error("json err", e);
                }
            }
        }
    }

    @Override
    public BidResponse getResponse(BidRequest bidRequest, Buyer buyer) {
        KeyGenerator key = buildKey(bidRequest, buyer);
        if (Objects.isNull(key)) {
            if (log.isDebugEnabled()) {
                log.debug("uid is null , fail to save response");
            }
            return null;
        }
        BidResponse bidResponse = fetcher.fetch(key, BidResponse.class);
        updater.remove(key);
        return bidResponse;
    }

    @Override
    public List<Deal> getCacheDealList( Map<Buyer, BidRequest> forCache) {
        List<Deal> dealList = new ArrayList<>();
        for (Map.Entry<Buyer, BidRequest> entry : forCache.entrySet()) {
            if (Objects.nonNull(entry)) {
                BidResponse bidResponse = getResponse(entry.getValue(), entry.getKey());
                if (Objects.nonNull(bidResponse)) {
                    Deal cache = new Deal(entry.getKey(), entry.getValue(), bidResponse);
                    dealList.add(cache);
                }
            }
        }
        return dealList;
    }

    @Override
    public void removeResponse(Deal deal) {
        if (Objects.nonNull(deal)) {
            KeyGenerator keyGenerator = buildKey(deal.getBidRequest(), deal.getBuyer());
            updater.remove(keyGenerator);
        }
    }
}
