package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.ingetration.AdxCounterServiceIntegration;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.filter.PriceLowerFilter;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @since 2019-12-12
 */
@Slf4j
@Configuration
public class RtbExchanger implements Exchanger {

    private final AdxProperties properties;
    private final AdxCounterServiceIntegration service;
    private final BuyerMeter buyerMeter;
    private final Counter counter;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final PriceLowerFilter priceLowerFilter;
    private final AdvertPointService advertPointService;
    private final DataLogDao dataLogDao;
    public RtbExchanger(AdxProperties properties, AdxCounterServiceIntegration service, BuyerMeter buyerMeter,
                        Counter counter, EventPark eventPark, BuyerResponseCache buyerResponseCache,
                        PriceLowerFilter priceLowerFilter, AdvertPointService advertPointService, DataLogDao dataLogDao) {
        this.properties = properties;
        this.service = service;
        this.buyerMeter = buyerMeter;
        this.counter = counter;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.priceLowerFilter = priceLowerFilter;
        this.advertPointService = advertPointService;
        this.dataLogDao = dataLogDao;
    }

    @Override
    public List<Deal> bidding(BidRequest bidRequest, Collection<BuyerWrapper> buyers) {
        int size = buyers.size();
        if (0 == size) {
            return Collections.emptyList();
        }
        if (null == bidRequest) {
            return Collections.emptyList();
        }

        ThreadPoolExecutor executor = new ThreadPoolExecutor(size, size, 5, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("exchange"));
        executor.allowCoreThreadTimeOut(true);

        List<Future<Deal>> futures = Lists.newArrayListWithCapacity(size);
        //等待时间内计数器
        CountDownLatch waitCountDown = new CountDownLatch(buyers.size());
        //超时时间内计数器
        CountDownLatch overtimeCountDown = new CountDownLatch(1);
        Map<Buyer, BidRequest> forCache = new HashMap<>();
        for (final BuyerWrapper buyer : buyers) {
            forCache.put(buyer.getBuyer(), bidRequest);
            if (!dspQPSLimit(buyer.getBuyer().getDsp())) {
                continue;
            }
            ExchangeParam exchangeParam = new ExchangeParam();
            exchangeParam.setDispatcher(buyer.getDispatcher());
            Future<Deal> future = executor.submit(new ExchangeCallable(buyer.getBuyer(), bidRequest, service, buyerMeter,
                    exchangeParam, eventPark, buyerResponseCache, waitCountDown, overtimeCountDown, advertPointService,
                    AdxContextFactory.get().getBidConfig(), dataLogDao));
            futures.add(future);
        }

        //处理等待时间期间逻辑
        try {
            //等待时间
            waitCountDown.await(bidRequest.getWaitTime(), TimeUnit.MILLISECONDS);
            //比价业务处理
            List<Deal> waitDeals = Lists.newArrayList();
            Iterator<Future<Deal>> futureIterator = futures.iterator();
            while (futureIterator.hasNext()) {
                Future<Deal> future = futureIterator.next();
                try {
                    if (future.isDone()) {
                        Deal response = future.get();
                        if (null != response) {
                            waitDeals.add(response);
                        }
                        futureIterator.remove();
                    }
                } catch (InterruptedException | ExecutionException e) {
                    // 超时或意外终止
                    future.cancel(true);
                    log.error("execute: ", e);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("wait deals = {}",
                        waitDeals.stream()
                                .map(Deal::getBuyer)
                                .map(Buyer::getDsp)
                                .map(Dsp::getDspCode)
                                .collect(Collectors.toList()).toString());
            }
            List<Deal> firstDeals = new ArrayList<>();
            List<Deal> waitCacheDeals = buyerResponseCache.getCacheDealList(forCache);
            if (log.isDebugEnabled()) {
                log.debug("wait deals size = {}, cache deals size = {}", waitDeals.size(), waitCacheDeals.size());
            }
            firstDeals.addAll(waitDeals);
            firstDeals.addAll(waitCacheDeals);
            if (!CollectionUtils.isEmpty(firstDeals)) {
                priceLowerFilter.filter(firstDeals);
                if (!CollectionUtils.isEmpty(firstDeals)) {
                    if (log.isDebugEnabled()) {
                        log.debug("first deals = {}",  firstDeals.stream()
                                .map(Deal::getBuyer)
                                .map(Buyer::getDsp)
                                .map(Dsp::getDspCode)
                                .collect(Collectors.toList()).toString());
                    }
                    Deal deal = firstDeals.iterator().next();
                    if (waitCacheDeals.contains(deal)) {
                        if (log.isDebugEnabled()) {
                            log.debug("use cache, remove cache");
                        }
                        buyerResponseCache.removeResponse(deal);
                    }
                    return firstDeals;
                }
            }

            if (futures.size() == 0) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }
            //处理超时时间期间逻辑
            //超时时间
            overtimeCountDown.await(bidRequest.getOvertime() - bidRequest.getWaitTime(), TimeUnit.MILLISECONDS);
            //比价业务处理
            List<Deal> overDeals = Lists.newArrayList();
            Iterator<Future<Deal>> secondFutureIterator = futures.iterator();
            while (secondFutureIterator.hasNext()) {
                Future<Deal> future = secondFutureIterator.next();
                try {
                    if (future.isDone()) {
                        Deal response = future.get();
                        if (null != response) {
                            overDeals.add(response);
                        }
                    }
                } catch (InterruptedException | ExecutionException e) {
                    // 超时或意外终止
                    future.cancel(true);
                    log.error("execute: ", e);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("overtime deals = {}",
                        overDeals.stream()
                                .map(Deal::getBuyer)
                                .map(Buyer::getDsp)
                                .map(Dsp::getDspCode)
                                .collect(Collectors.toList()).toString());
            }
            List<Deal> secondDeals = new ArrayList<>();
            secondDeals.addAll(overDeals);
            priceLowerFilter.filter(secondDeals);
            if (!CollectionUtils.isEmpty(secondDeals)) {
                if (log.isDebugEnabled()) {
                    log.debug("second deals = {}",
                            secondDeals.stream()
                                    .map(Deal::getBuyer)
                                    .map(Buyer::getDsp)
                                    .map(Dsp::getDspCode)
                                    .collect(Collectors.toList()).toString());
                }
                return secondDeals;
            }
        } catch (InterruptedException e) {
            log.error("err", e);
        }
        //最后超时抛出异常不参与竞价
        throw new AdxException(AdxErrEnum.DSP_TIMEOUT);
    }

    /**
     * DSP-QPS限制
     * @param dsp
     * @return
     */
    private boolean dspQPSLimit(Dsp dsp) {
        if (dsp == null) {
            return false;
        }
        Integer qps = dsp.getQps();
        if (qps == null) {
            return true;
        }
        if (qps.intValue() < 0) {
            return true;
        }
        if (qps.intValue() == 0) {
            return false;
        }
        long time = LocalDateTime.now().toEpochSecond(ZoneOffset.of("+8"));
        KeyGenerator keyGenerator = CacheKeys.getDspNowQPS(time + "");
        Long now = counter.hincrementAndGet(keyGenerator, dsp.getDspCode(), 600);
        if (now > qps) {
//            counter.hincrementAndGet(CacheKeys.getDspQPSBan(time+""), dsp.getDspCode(), 600);
            return false;
        } else {
            return true;
        }
    }
}
