package com.bxm.adx.common.sell;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.buyers.PriorityBuyers;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.optimization.Optimization;
import com.bxm.adx.common.buy.optimization.OptimizationDao;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.openlog.event.internal.MediaRequestEvent;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.user.id.generator.DeviceHelper;
import com.bxm.user.id.generator.DeviceInfo;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author allen
 * @since 2019-12-20
 */
@Slf4j
@Configuration
public class DefaultBidRequestWrapper implements BidRequestWrapper {

    private final PriorityBuyers priorityBuyers;
    private final PositionService positionService;
    private final AdxProperties properties;
    private final OptimizationDao optimizationDao;
    private final IpService ipService;
    private final EventPark eventPark;

    public DefaultBidRequestWrapper(PriorityBuyers priorityBuyers, PositionService positionService,
                                    AdxProperties properties,
                                    OptimizationDao optimizationDao, IpService ipService, EventPark eventPark) {
        this.priorityBuyers = priorityBuyers;
        this.positionService = positionService;
        this.properties = properties;
        this.optimizationDao = optimizationDao;
        this.ipService = ipService;
        this.eventPark = eventPark;
    }

    @Override
    public MarketOrders packing(BidRequest bidRequest) {
        if (null == bidRequest) {
            return null;
        }
        if (Objects.isNull(bidRequest.getId())) {
            bidRequest.setId(RandomStringUtils.randomAlphabetic(10));
        }
        AdxContextFactory.get().setBidRequest(bidRequest);
        BidConfig bidConfig = new BidConfig().setDisDot(Math.abs(bidRequest.getId().hashCode() % 100) >= properties.getDotSimplingPercent());
        AdxContextFactory.get().setBidConfig(bidConfig);

        List<Impression> imps = bidRequest.getImps();
        if (CollectionUtils.isEmpty(imps)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        Map<Position, List<List<BuyerWrapper>>> positionListMap = Maps.newHashMap();
        List<Position> positionList = new ArrayList<>();
        // 获取所有的广告位
        for (Impression imp : imps) {
            String tag_id = imp.getTag_id();
            if (StringUtils.isBlank(tag_id)) {
                continue;
            }
            Position position = positionService.getByPositionId(tag_id);
            if (null != position) {
                AdxContextFactory.get().setPosition(position);

                imp.setTag_id(position.getPositionId());
                positionList.add(position);
                bidRequest.setCooperationType(position.getCooperationType());
                Integer dockingMethodType = position.getDockingMethodType();
                bidRequest.setDockingMethodType(dockingMethodType);
                bidRequest.setBxmTagId(position.getPositionId());
                Long waitTime = position.getWaitTime();
                Long overTime = position.getOvertime();
                if (Objects.nonNull(dockingMethodType)) {
                    if (Objects.isNull(waitTime)) {
                        if (dockingMethodType == AdxConstants.DockingMethodType.API) {
                            waitTime = AdxConstants.API_WAIT_TIME;
                        } else {
                            waitTime = AdxConstants.SDK_WAIT_TIME;
                        }
                    }
                    if (Objects.isNull(overTime)) {
                        if (dockingMethodType == AdxConstants.DockingMethodType.API) {
                            overTime = AdxConstants.API_OVER_TIME;
                        } else {
                            overTime = AdxConstants.SDK_OVER_TIME;
                        }
                    }
                } else {
                    waitTime = 0L;
                    overTime = 0L;
                }
                bidRequest.setWaitTime(waitTime);
                bidRequest.setOvertime(overTime);
            }
            if (null != position && position.isEnabled()) {
                // 查找这些广告位分配的卖家和等级
                List<List<BuyerWrapper>> buyers = priorityBuyers.findAsPriority(position, imp.getBid_floor(), imp.getBid_top());
                //处理广告优化
                handleOptimization(position, bidRequest, buyers);
                handleBackupEntrance(bidRequest, buyers);
                positionListMap.put(position, buyers);
            }
        }

        if (CollectionUtils.isEmpty(positionListMap)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        //埋点媒体请求
        eventPark.post(new MediaRequestEvent(this, bidRequest, bidConfig));

        // 索引即表示优先级，从0开始。
        List<MarketRequest> requests = Lists.newArrayListWithCapacity(properties.getMaxCapacityOfBuyers());
        // 重组，分出每一个优先级中对应的DSP+广告位
        for (Map.Entry<Position, List<List<BuyerWrapper>>> positionListEntry : positionListMap.entrySet()) {
            Position position = positionListEntry.getKey();
            List<List<BuyerWrapper>> list = positionListEntry.getValue();
            int level = list.size();
            if (level == 0) {
                // 没有设置分流策略
                continue;
            }

            for (int i = 0; i < level; i++) {
                // 这个广告位第i级的DSP配置列表
                List<BuyerWrapper> array = list.get(i);
                if (CollectionUtils.isEmpty(array)) {
                    continue;
                }
                MarketRequest marketRequest = CollectionUtils.isEmpty(requests) ? null : requests.size() > i ? requests.get(i) : null;
                if (null == marketRequest) {
                    marketRequest = new MarketRequest(i);
                    requests.add(i, marketRequest);
                }
                marketRequest.addBuyers(array);
                marketRequest.addPosition(position);
            }
        }

        MarketOrders marketOrders = new MarketOrders(bidRequest, requests, positionList, getUid(bidRequest));
        if (log.isDebugEnabled()) {
            log.debug("{}", JsonHelper.convert(marketOrders));
        }
        return marketOrders;
    }

    private void handleBackupEntrance(BidRequest request, List<List<BuyerWrapper>> buyers) {
        String entanceName = request.getEntrance_name();
        if (StringUtils.isEmpty(entanceName)) {
            return;
        }
        if ("backup".equalsIgnoreCase(entanceName)) {
            buyers.clear();
            List<BuyerWrapper> buyerList = new ArrayList<>();
            buyers.add(buyerList);
            Map<String, Buyer> allBuyers = priorityBuyers.getAllBuyers();
            Buyer buyer = allBuyers.get("pangu-backup");
            if (buyer == null) {
                throw new AdxException("can't find backup-dsp ");
            }
            Dispatcher dispatcher = new Dispatcher();
            dispatcher.setDspBasePrice(BigDecimal.ZERO);
            dispatcher.setDspId(buyer.getDsp().getId());
            dispatcher.setDspAvgPrice(BigDecimal.ZERO);
            dispatcher.setProfitMargin(new BigDecimal(100));
            buyerList.add(new BuyerWrapper(buyer, dispatcher));
        }
    }

    private void handleOptimization(Position position, BidRequest request, List<List<BuyerWrapper>> buyers) {
        Optimization optimization = optimizationDao.getOptimizationByPositionId(position.getPositionId());
        if (Objects.isNull(optimization)) {
            return;
        }
        String regionCodes = optimization.getRegion();
        if (!org.springframework.util.StringUtils.isEmpty(regionCodes)) {
            IP ipp = getIp(request);
            Set<String> set = Sets.newHashSet();
            set.addAll(Arrays.asList(regionCodes.split(",")));
            if (Objects.isNull(ipp) || !ipp.in(set)) {
                return;
            }
        }

        Map<String, Buyer> allBuyers = priorityBuyers.getAllBuyers();
        Buyer buyer = allBuyers.get("optimization");
        if (buyer == null) {
            throw new AdxException("can't find op-dsp ");
        }
        Dispatcher dispatcher = new Dispatcher();
        dispatcher.setDspPosid(optimization.getDspPositionId());
        buyers.add(Lists.newArrayList(new BuyerWrapper(buyer, dispatcher)));
        AdxContextFactory.get().setOptimization(true);
    }

    /**
     * 统一方法获取uid
     *
     * @param bidRequest
     * @return
     */
    private String getUid(BidRequest bidRequest) {
        Device device = bidRequest.getDevice();
        if (device == null) {
            return null;
        }
        DeviceInfo info = new DeviceInfo()
                .setImei(device.getImei())
                .setImeiMd5(device.getImei_md5())
                .setOaid(device.getOaid())
                .setOaidMd5(device.getOaid_md5())
                .setAnid(device.getDpid())
                .setAnidMd5(device.getDpid_md5())
                .setIdfa(device.getIdfa())
                .setIdfaMd5(device.getIdfa_md5())
                .setOs(device.isAndroid() ? DeviceInfo.OS_ANDROID : device.isIos() ? DeviceInfo.OS_IOS : DeviceInfo.OS_UNKNOWN);
        return DeviceHelper.getUid(info);
    }

    private IP getIp(BidRequest bidRequest) {
        IP ipp = null;
        if (bidRequest != null) {
            Device device = bidRequest.getDevice();
            if (device != null) {
                String ip = device.getIp();
                if (!org.springframework.util.StringUtils.isEmpty(ip)) {
                    ipp = ipService.analyze(ip);
                    AdxContextFactory.get().setIp(ipp);
                }
            }
        }
        return ipp;
    }
}
