package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 低价过滤器，同一个广告位只保留出价最高的。
 * 只支持单个seatbid
 *
 * @author allen
 * @since 2019-12-18
 */
@Order(Ordered.LOWEST_PRECEDENCE)
@Slf4j
@Configuration
public class PriceLowerFilter implements Filter {

    @Override
    public void filter(List<Deal> deals) {
        Map<String, BigDecimal> higher = Maps.newHashMap();
        String key = "higher";
        deals.removeIf(deal -> {
            BidResponse response = deal.getBidResponse();
            if (null == response) {
                return true;
            }
            List<SeatBid> seatBids = response.getSeat_bid();
            if (CollectionUtils.isEmpty(seatBids)) {
                return true;
            }
            seatBids.removeIf(seatBid -> {
                List<Bid> bidList = seatBid.getBid();
                if (CollectionUtils.isEmpty(bidList)) {
                    return true;
                }
                bidList.removeIf(b -> {
                    BigDecimal price = b.getPrice();
                    if (null == price) {
                        return true;
                    }

                    BidRequest request = deal.getBidRequest();
                    BigDecimal bidFloor = null;
                    if (Objects.nonNull(request.getDockingMethodType())
                            && DockingMethodTypeEnum.API_OPERATION.getType() == request.getDockingMethodType()) {
                        bidFloor = request.getImps().iterator().next().getBid_floor();
                    }

                    if (Objects.nonNull(bidFloor)) {
                        if (price.compareTo(bidFloor) < 0) {
                            return true;
                        }
                    }

                    BigDecimal max = null;
                    if (org.springframework.util.CollectionUtils.isEmpty(higher)) {
                        MapHelper.get(higher, key, price);
                        return false;
                    } else {
                        max = MapHelper.get(higher, key, price);
                        if (price.compareTo(max) > 0) {
                            higher.put(key, price);
                            return false;
                        } else {
                            if (log.isDebugEnabled()) {
                                log.debug("price filter, dsp = {}", deal.getBuyer().getDsp().getDspCode());
                            }
                            return true;
                        }
                    }
                });
                return CollectionUtils.isEmpty(bidList);
            });
            return CollectionUtils.isEmpty(seatBids);
        });
    }

    public static void main(String[] args) {
        System.out.println(Float.compare(0, 0));
        System.out.println(Float.compare(1, 0));
        System.out.println(Float.compare(0, 1));
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }
}
