package com.bxm.adx.common.buy.dispatcher;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfig;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfigCaching;
import com.bxm.adx.common.buy.dispatcher.abtest.cached.DispatcherCached;
import com.bxm.adx.common.buy.dispatcher.abtest.cached.DispatcherConfigCached;
import com.bxm.adx.common.caching.Rule;
import com.bxm.adx.common.creative.replace.ReplaceCreative;
import com.bxm.adx.common.creative.replace.ReplaceCreativeDao;
import com.bxm.adx.common.rule.WhiteBlackSetRule;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.utils.AdxUtils;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @since 2019-12-16
 */
@Configuration
@Slf4j
public class DispatcherServiceImpl implements DispatcherService {

    private final ReplaceCreativeDao replaceCreativeDao;
    private final DispatcherConfigCached dispatcherConfigCached;
    private final DispatcherABConfigCaching dispatcherABConfigCaching;
    private final DispatcherCached dispatcherCached;
    private final Fetcher fetcher;
    private final Updater updater;

    public DispatcherServiceImpl(ReplaceCreativeDao replaceCreativeDao,
                                 DispatcherConfigCached dispatcherConfigCached, DispatcherABConfigCaching dispatcherABConfigCaching,
                                 DispatcherCached dispatcherCached, Fetcher fetcher, Updater updater) {
        this.replaceCreativeDao = replaceCreativeDao;
        this.dispatcherConfigCached = dispatcherConfigCached;
        this.dispatcherABConfigCaching = dispatcherABConfigCaching;
        this.dispatcherCached = dispatcherCached;
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public Map<Integer, Collection<Dispatcher>> getPriority(Position position, DispatcherParam param) {
        Collection<Dispatcher> dispatchers = filterDispatcher(position, param);
        Map<Integer, Collection<Dispatcher>> dsps = Maps.newTreeMap();
        if (CollectionUtils.isEmpty(dispatchers)) {
            return dsps;
        }

        dispatchers = dispatchers.stream()
                .filter(dispatcherDspCacheVO -> dispatcherDspCacheVO.getOpened() == Dispatcher.DISPATCHER_OPENED_YES)
                .filter(dispatcherDspCacheVO -> limitedByPrice(dispatcherDspCacheVO, param))
                .filter(dispatcherDspCacheVO -> limitedByAppPackageName(dispatcherDspCacheVO, param))
                .filter(dispatcherDspCacheVO -> limitedByInstallAppList(dispatcherDspCacheVO, param))
                .collect(Collectors.toList());

        for (Dispatcher dispatcher : dispatchers) {
            int priority = dispatcher.getPriority();

            Collection<Dispatcher> collection = MapHelper.get(dsps, priority, new ArrayList<>());

            //添加需要替换的素材
            List<Long> creativeIds = dispatcher.getCreativeIdList();
            if (CollectionUtils.isNotEmpty(creativeIds)) {
                List<ReplaceCreative> creativeList = replaceCreativeDao.queryCreativeListByIds(creativeIds);
                dispatcher.setCreativeList(creativeList);
            }

            dispatcher.setDispatcherPriceConfig(getPriceConfigByTime(dispatcher.getDispatcherDspPosPriceConfigVOS()));
            collection.add(dispatcher);
        }
        return dsps;
    }

    /**
     * 获取广告位流量分配信息
     *
     * @param position
     * @return
     */
    private Collection<Dispatcher> filterDispatcher(Position position, DispatcherParam dispatcherParam) {
        String positionId = position.getPositionId();
        DispatcherABConfig dispatcherABConfig = dispatcherABConfigCaching.get(positionId);
        if (Objects.isNull(dispatcherABConfig)) {
            return null;
        }

        Integer dockingMethodType = position.getDockingMethodType();
        //先进行用户分桶
        String uid = AdxContextFactory.get().getUid();
        if (StringUtils.isEmpty(uid)) {
            uid = RandomStringUtils.randomAlphabetic(13);
        }
        int userBucket = AdxUtils.getBucket(uid);
        //根据分桶拿到对应的流量分配配置id
        Long configId = dispatcherConfigCached.getConfigId(positionId, userBucket);
        Collection<Dispatcher> dispatchers = null;
        //如果分桶失败，则使用旧分配数据
        if (Objects.isNull(configId)) {
            if (log.isWarnEnabled()) {
                log.warn("position {} user bucket {} fail ", positionId, userBucket);
            }
            return null;

        }
        AdxContextFactory.get().setConfigId(configId);
        //API广告位不支持重排序
        if (Objects.nonNull(dockingMethodType) && dockingMethodType == AdxConstants.DockingMethodType.API) {
            return dispatcherCached.getDispatchers(positionId, configId);
        }
        //获取上一次排序结果
        dispatchers = fetcher.hfetchList(CacheKeys.Dispather.getDispatcherKeyGeneratorByUid(positionId), uid, Dispatcher.class);
        //如果没拿到上一次结果，则重新根据配置ID拿到流量分配数据
        if (CollectionUtils.isEmpty(dispatchers)) {
            dispatchers = dispatcherCached.getDispatchers(positionId, configId);
        }
        //根据最新频次状况重排序
        dispatchers = reorder(dispatchers, uid, dispatcherParam);
        //记录重排序结果给下一次用，当天24点过期
        LocalDateTime midnight = LocalDateTime.now().plusDays(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        long seconds = ChronoUnit.SECONDS.between(LocalDateTime.now(), midnight);
        updater.hupdate(CacheKeys.Dispather.getDispatcherKeyGeneratorByUid(positionId), uid, dispatchers, (int) seconds);

        return dispatchers;
    }

    /**
     * 获取已用频次
     *
     * @param dispatcher
     * @param uid
     * @return
     */
    private int getPositionDspFreq(Dispatcher dispatcher, String uid) {
        KeyGenerator keyGenerator = getFreqKeyGenerator(dispatcher, uid);
        Long freq = fetcher.fetch(keyGenerator, Long.class);
        return Objects.isNull(freq) ? 0 : freq.intValue();
    }

    /**
     * 重置已用频次
     *
     * @param dispatcher
     * @param uid
     */
    private void resetPositionDspFreq(Dispatcher dispatcher, String uid) {
        KeyGenerator keyGenerator = getFreqKeyGenerator(dispatcher, uid);
        updater.update(keyGenerator, 0);
    }

    /**
     * 获取频次rediskey
     *
     * @param dispatcher
     * @param uid
     * @return
     */
    private KeyGenerator getFreqKeyGenerator(Dispatcher dispatcher, String uid) {
        return AdxKeyGenerator.Counter.getPositionDspFrequency(dispatcher.getPositionId(),
                dispatcher.getConfigId().toString(), dispatcher.getDspId().toString(), uid);
    }

    /**
     * 根据频次重排序
     *
     * @param ds
     * @param uid
     * @return
     */
    private Collection<Dispatcher> reorder(Collection<Dispatcher> ds, String uid, DispatcherParam dispatcherParam) {
        //并发模式不重排序
        if (dispatcherParam.isSDKConcurrentModel()) {
            return ds;
        }
        List<Dispatcher> result = new ArrayList<>();
        //获取排好序的优先级
        List<Integer> ps = ds.stream().map(Dispatcher::getPriority).sorted().distinct().collect(Collectors.toList());
        //获取最大优先级max，达到频次的dsp的优先级调整为max+1
        Integer max = ps.get(ps.size() - 1);
        for (Integer p : ps) {
            //获取当前优先级状态正常的dsp
            List<Dispatcher> dispatchers = ds.stream()
                    .filter(dispatcherDspCacheVO -> dispatcherDspCacheVO.getPriority() == p)
                    .filter(dispatcherDspCacheVO -> dispatcherDspCacheVO.getOpened() == 1)
                    .collect(Collectors.toList());
            //当同优先级有多个dsp时频次重排序不起效，不调整优先级
            if (dispatchers.size() == 1) {
                Dispatcher dc = dispatchers.iterator().next();
                Integer initFreq = dc.getFrequency();//设置的频次
                if (null != initFreq) {//频次为null则视为无限频次，无限频次不需要重排序
                    Integer used = getPositionDspFreq(dc, uid);//已用频次
                    if (used >= initFreq) {//频次已经用完
                        if (dc.getPriority() != max.intValue()) {//如果当前分配优先级就是最大级则不用重置优先级
                            max = max + 1;
                            dc.setPriority(max);//调整dsp优先级，并调整当前最大优先级
                        }
                        resetPositionDspFreq(dc, uid);
                    }
                }
                result.add(dc);
            } else {
                result.addAll(dispatchers);
            }
        }
        return result;
    }

    /**
     * 平均cpm价格过滤
     *
     * @param dispatcher
     * @param param
     * @return
     */
    private boolean limitedByPrice(Dispatcher dispatcher, DispatcherParam param) {
        if (param.isSDKConcurrentModel()) {
            BigDecimal price = dispatcher.getDspAvgPrice();
            if (price.compareTo(param.getMinPrice().movePointLeft(2)) == -1) {
                if (log.isDebugEnabled()) {
                    log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "< min");
                }
                return false;
            }
            if (price.compareTo(param.getMaxPrice().movePointLeft(2)) >= 0) {
                if (log.isDebugEnabled()) {
                    log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "> max");
                }
                return false;
            }
        }
        return true;
    }

    /**
     * app已安装包名定向
     *
     * @param dispatcher
     * @param param
     * @return
     */
    private boolean limitedByInstallAppList(Dispatcher dispatcher, DispatcherParam param) {
        List<String> installList = param.getInstallAppList();
        Rule rule = dispatcher.getInstalledAppDirect();
        if (Objects.isNull(rule)) {
            return true;
        }
        WhiteBlackSetRule setRule = new WhiteBlackSetRule(rule);
        Set<String> set = setRule.getSet();
        if (setRule.isWhite()) {
            if (CollectionUtils.isEmpty(installList)) {
                if (log.isDebugEnabled()) {
                    log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "installApp-white-empty");
                }
                return false;
            } else {
                Collection<String> result = CollectionUtils.intersection(installList, set);
                if (CollectionUtils.isNotEmpty(result)) {
                    return true;
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "installApp-white-empty");
                    }
                    return false;
                }
            }
        } else {
            if (CollectionUtils.isEmpty(installList)) {
                return true;
            } else {
                Collection<String> result = CollectionUtils.intersection(installList, set);
                if (CollectionUtils.isNotEmpty(result)) {
                    if (log.isDebugEnabled()) {
                        log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "installApp-black");
                    }
                    return false;
                } else {
                    return true;
                }
            }
        }
    }

    /**
     * 媒体app包名定向
     *
     * @param dispatcher
     * @param param
     * @return
     */
    private boolean limitedByAppPackageName(Dispatcher dispatcher, DispatcherParam param) {
        String packageName = param.getMediaAppPackageName();
        Rule rule = dispatcher.getMediaAppDirect();
        if (Objects.isNull(rule)) {
            return true;
        }

        WhiteBlackSetRule setRule = new WhiteBlackSetRule(rule);
        Set<String> set = setRule.getSet();
        if (setRule.isWhite()) {
            if (StringUtils.isEmpty(packageName)) {
                if (log.isDebugEnabled()) {
                    log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "packageName-white-empty");
                }
                return false;
            } else {

                if (set.contains(packageName)) {
                    return true;
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "packageName-white");
                    }
                    return false;
                }
            }
        } else {
            if (StringUtils.isEmpty(packageName)) {
                return true;
            } else {
                if (set.contains(packageName)) {
                    if (log.isDebugEnabled()) {
                        log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "packageName-black");
                    }
                    return false;
                } else {
                    return true;
                }
            }
        }
    }

    private DispatcherPriceConfig getPriceConfigByTime(List<DispatcherPriceConfig> configs) {
        if (CollectionUtils.isEmpty(configs)) {
            return null;
        }
        int hour = LocalDateTime.now().get(ChronoField.HOUR_OF_DAY);
        for (DispatcherPriceConfig config : configs) {
            Integer start = config.getStartTime();
            Integer end = config.getEndTime();
            if (Objects.nonNull(start) && Objects.nonNull(end)) {
                if (start <= hour && hour < end) {
                    return config;
                }
            }
        }

        return DispatcherPriceConfig.builder()
                .ltPriceDiscountFactor(BigDecimal.ONE)
                .gtPriceDiscountFactor(BigDecimal.ONE)
                .biddingCoefficient(BigDecimal.ONE)
                .profitMargin(BigDecimal.ZERO)
                .build();
    }
}
