package com.bxm.adx.common.market.exchange;

import com.alibaba.fastjson.JSONException;
import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.log.dsplog.DspLogRecord;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.exchange.rebuild.response.AdxBidResponseBuildFactory;
import com.bxm.adx.common.market.exchange.rebuild.response.ResponseBuildAttribute;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.openlog.event.internal.AdxRequestEvent;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.rebuild.request.RebuildRequestChain;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;

/**
 * @author allen
 * @date 2020-10-30
 * @since 1.0
 */
@Slf4j
public class ExchangeCallable implements Callable<Deal> {

    private final Buyer buyer;
    private final Dispatcher dispatcher;
    private final BidRequest bidRequest;
    private final BidRequest adxBidRequest;
    private final BuyerMeter buyerMeter;
    private final ExchangeParam exchangeParam;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final CountDownLatch waitCountDown;
    private final CountDownLatch overtimeCountDown;
    private final BidConfig bidConfig;
    private final DataLogDao dataLogDao;
    private final DspLogRecord dspLogRecord;
    private final RebuildRequestChain requestChain;
    private final AdxBidResponseBuildFactory responseBuildFactory;

    public ExchangeCallable(BuyerWrapper buyer, BidRequest bidRequest, BidRequest adxBidRequest,
                            BuyerMeter buyerMeter, ExchangeParam exchangeParam, EventPark eventPark,
                            BuyerResponseCache buyerResponseCache,
                            CountDownLatch waitCountDown, CountDownLatch overtimeCountDown,
                            BidConfig bidConfig, DataLogDao dataLogDao, DspLogRecord dspLogRecord,
                            RebuildRequestChain requestChain, AdxBidResponseBuildFactory responseBuildFactory) {

        this.buyer = buyer.getBuyer();
        this.dispatcher = buyer.getDispatcher();
        this.bidRequest = bidRequest;
        this.adxBidRequest = adxBidRequest;
        this.buyerMeter = buyerMeter;
        this.exchangeParam = exchangeParam;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.waitCountDown = waitCountDown;
        this.overtimeCountDown = overtimeCountDown;
        this.bidConfig = bidConfig;
        this.dataLogDao = dataLogDao;
        this.dspLogRecord = dspLogRecord;
        this.requestChain = requestChain;
        this.responseBuildFactory = responseBuildFactory;
    }

    @Override
    public Deal call() {
        long startTime = System.currentTimeMillis();

        String name = buyer.getCode();
        BuyModelAdapter modelAdapter = buyer.getModelAdapter();
        if (null == modelAdapter) {
            log.warn("BuyModelAdapter [{}] not found!", name);
            return null;
        }

        byte[] request = null;

        Deal deal = null;
        try {
            // 对上下文设置必要的对象
            ExchangeContext.putBidRequest(bidRequest);
            ExchangeContext.putBuyer(buyer);
            ExchangeContext.putDispatch(dispatcher);

            rebuildAdxBidRequest();
            dataLog();

            ExchangeContext.putAdxRequest(adxBidRequest);
            request = modelAdapter.buildRequest(adxBidRequest);
            if (ArrayUtils.isEmpty(request)) {
                return null;
            }
            Dsp dsp = buyer.getDsp();
            String configId = StringUtils.EMPTY;
            if (Objects.nonNull(dispatcher) && Objects.nonNull(dispatcher.getConfigId())) {
                configId = dispatcher.getConfigId().toString();
            }
            //埋点adx请求
            eventPark.post(new AdxRequestEvent(this, bidRequest, adxBidRequest, dsp.getId().toString(), configId,
                    bidConfig));

            // 请求指标
            buyerMeter.increaseRequest(buyer);

            byte[] response = offer(request);
            if (ArrayUtils.isEmpty(response)) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }

            deal = new Deal(buyer, request, response, bidRequest, adxBidRequest);

            if (!deal.isBidSuccess()) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }
            //记录有广告返回的日志
            dspLogRecord.dspLog(bidRequest, deal.getBidResponse(), dispatcher);
            //填充response
            rebuildResponse(deal);

            //埋点dsp返回
            eventPark.post(new DspBidEvent(this, bidRequest, adxBidRequest, deal.getBidResponse(),
                    dsp.getId().toString(), bidConfig));

            // 填充指标
            buyerMeter.increasePadding(buyer);

            return deal;
        } catch (Exception e) {
            AdxErrEnum errEnum = dealException(e);
            return new Deal(buyer, request, bidRequest, adxBidRequest, errEnum);
        } finally {
            long endTime = System.currentTimeMillis();
            long interval = endTime - startTime;
            if (log.isDebugEnabled()) {
                log.debug("buy = {}, interval = {}", JsonHelper.convert(buyer), interval);
            }
            if (interval < bidRequest.getWaitTime()) {
                waitCountDown.countDown();
            } else if (interval < bidRequest.getOvertime()) {
                overtimeCountDown.countDown();
            } else {
                if (Objects.nonNull(deal) && deal.isBidSuccess()) {
                    buyerResponseCache.saveResponse(deal.getAdxRequest(), deal.getBidResponse(), buyer);
                }
            }
            ExchangeContext.remove();
        }
    }

    private byte[] offer(byte[] request) {
        long start = System.nanoTime();
        byte[] response;
        try {
            response = buyer.offer(request);
        } finally {
            // 耗时指标
            buyerMeter.recordRequest(buyer, start);
        }
        return response;
    }

    /**
     * 异常处理
     *
     * @param e
     * @return
     */
    private AdxErrEnum dealException(Exception e) {
        AdxErrEnum adxErrEnum = null;
        if (e instanceof AdxException) {
            AdxException adxException = (AdxException) e;
            adxErrEnum = adxException.getAdxErrEnum();
        } else if (e instanceof JSONException) {

        } else {
            if (log.isErrorEnabled()) {
                log.error("deal call err", e);
            }
        }
        adxErrEnum = Optional.ofNullable(adxErrEnum).orElse(AdxErrEnum.UNKNOWN_ERR);
        return adxErrEnum;
    }

    /**
     * 重构adx-bidrequest
     *
     */
    private void rebuildAdxBidRequest() {
        requestChain.handler(dispatcher, adxBidRequest);
    }

    /**
     * 请求响应log
     */
    private void dataLog() {
        for (Impression impression : bidRequest.getImps()) {
            ExchangeContext.putDataLog(dataLogDao.existDataLog(impression.getTag_id(), dispatcher.getDspPosid()));
        }
    }

    /**
     * 重构Response
     *
     * @param deal
     */
    private void rebuildResponse(Deal deal) {
        if (deal.isOp()) {
            return;
        }
        ResponseBuildAttribute attribute = ResponseBuildAttribute.builder()
                .bidRequest(deal.getBidRequest())
                .dispatcher(dispatcher)
                .build();
        responseBuildFactory.buildAdxBidResponse(deal.getBidResponse(), attribute);
    }
}
