package com.bxm.adx.common.buy.buyer;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.buy.dsp.DspChangedEvent;
import com.bxm.adx.common.utils.MapHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author allen
 * @date 2020-11-12
 * @since 1.0
 */
@Slf4j
@Configuration
public class DefaultHttpClientBuilder implements HttpClientBuilder, ApplicationListener<DspChangedEvent>, DisposableBean {

    private final AdxProperties properties;
    private final ConcurrentHashMap<String, CloseableHttpClient> httpClients = new ConcurrentHashMap<>();

    public DefaultHttpClientBuilder(AdxProperties properties) {
        this.properties = properties;
    }

    @Override
    public void onApplicationEvent(DspChangedEvent event) {
        Dsp dsp = event.getDsp();
        if (Objects.isNull(dsp)) {
            return;
        }
        String dspCode = dsp.getDspCode();
        if (httpClients.contains(dspCode)) {
            httpClients.put(dspCode, build0(dsp));
        } else {
            if (log.isInfoEnabled()) {
                log.info("[{}] Needn't Rebuild HttpClient instance.",
                        dsp.getDspCode());
            }
        }
    }

    @Override
    public void destroy() throws Exception {
    }

    @Override
    public HttpClient build(Dsp dsp) {
        if (Objects.isNull(dsp)) {
            return null;
        }
        String dspCode = dsp.getDspCode();
        return MapHelper.get(httpClients, dspCode, new MapHelper.InitializingValue<CloseableHttpClient>() {
            @Override
            public CloseableHttpClient initializing() {
                return build0(dsp);
            }
        });
    }

    private CloseableHttpClient build0(Dsp dsp) {
        int connectTimeOutInMillis = properties.getBuyerBiddingConnectTimeOutInMillis();
        int socketTimeOutInMillis = properties.getBuyerBiddingSocketTimeOutInMillis();

        if (Objects.nonNull(dsp)) {
            connectTimeOutInMillis = Objects.nonNull(dsp.getConnectTimeOut()) ? dsp.getConnectTimeOut() : connectTimeOutInMillis;
            socketTimeOutInMillis = Objects.nonNull(dsp.getSocketTimeOut()) ? dsp.getSocketTimeOut() : socketTimeOutInMillis;
        }

        final SocketConfig socketConfig = SocketConfig.custom().build();

        final PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setDefaultSocketConfig(socketConfig);
        connectionManager.setDefaultMaxPerRoute(properties.getBuyerDefaultMaxPerRoute());
        connectionManager.setMaxTotal(properties.getBuyerMaxTotal());

        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(connectTimeOutInMillis)
                .setCookieSpec(properties.getBuyerCookieSpec())
                .setSocketTimeout(socketTimeOutInMillis)
                .setConnectionRequestTimeout(100)
                .build();

        if (log.isInfoEnabled()) {
            log.info("[{}] Rebuild HttpClient instance. connectTimeOut={}, socketTimeOut={}",
                    dsp.getDspCode(), connectTimeOutInMillis, socketTimeOutInMillis);
        }
        return org.apache.http.impl.client.HttpClientBuilder.create()
                .evictExpiredConnections()
                .setDefaultRequestConfig(config)
                .disableAutomaticRetries()
                .setConnectionManager(connectionManager)
                .build();
    }
}
