package com.bxm.adx.common.buy.dispatcher.filter;

import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.DispatcherFlowProperties;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.common.buy.dispatcher.DispatcherFlowControl;
import com.bxm.adx.common.caching.Id;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.App;
import com.bxm.adx.common.utils.DateUtils;
import com.bxm.warcar.cache.KeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.core.Ordered;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 砍量过滤器
 * 务必放在最后执行
 *
 * @author fgf
 * @date 2023/5/4
 **/
@Slf4j
public class FlowControlFilter implements DispatcherFilter<Dispatcher> {

    private final DispatcherFlowControl dispatcherFlowControl;

    public FlowControlFilter(DispatcherFlowControl dispatcherFlowControl) {
        this.dispatcherFlowControl = dispatcherFlowControl;
    }

    @Override
    public void filter(DispatcherContext<Dispatcher> context, Set<Id> trash) {
        Collection<Dispatcher> dispatchers = context.getValues();
        BidRequest request = context.getRequest();
        String date = LocalDate.now().format(DateUtils.FORMAT_SIMPLE);
        Set<Dispatcher> removes = dispatchers.stream().filter(dispatcher -> {
            Byte flow = dispatcher.getChopQuantitySwitch();
            if (Objects.nonNull(flow) && Dispatcher.FLOW_OPENED_YES == flow) {
                KeyGenerator key = getKeyGenerator(date, dispatcher, request);
                Integer percent = dispatcherFlowControl.getPercent(key);
                if (Objects.nonNull(percent)) {
                    percent = fixPercent(key, percent);
                    int random = RandomUtils.nextInt(0, 100);
                    if (random < percent) {
                        return true;
                    }
                }
            }
            return false;
        }).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(removes)) {
            trash.addAll(removes);
        }
    }

    private Integer fixPercent(KeyGenerator keyGenerator, Integer percent) {
        if (percent.compareTo(DispatcherFlowProperties.CEILING) > 0) {
            log.warn("key {}, val {}", keyGenerator.generateKey(), percent);
            return DispatcherFlowProperties.CEILING;
        }
        if (percent.compareTo(DispatcherFlowProperties.FLOOR) < 0) {
            log.warn("key {}, val {}", keyGenerator.generateKey(), percent);
            return DispatcherFlowProperties.FLOOR;
        }
        return percent;
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    private KeyGenerator getKeyGenerator(String date, Dispatcher dispatcher, BidRequest request) {
        String dspId = dispatcher.getDspId().toString();
        String dspTagId = dispatcher.getDspPosid();
        String dspAppId = dispatcher.getDspAppid();
        String tagId = request.getImps().iterator().next().getTag_id();
        App app = request.getApp();
        String appPkg = "";
        if (Objects.nonNull(app)) {
            appPkg = Optional.ofNullable(app.getBundle()).orElse("");
        }

        return CacheKeys.DispatcherFlow.getDispatcherFlowPercent(date, dspId, dspAppId, dspTagId, tagId, appPkg);
    }
}
