package com.bxm.adx.common.caching;

import com.bxm.warcar.integration.listeners.AbstractNameBeanBus;
import lombok.extern.slf4j.Slf4j;

import java.util.Objects;

/**
 * <p>对象缓存，并包装更新处理器工厂，内部实现了接收到缓存更新到调用处理器进行处理的逻辑。</p>
 * <p>从 {@link AbstractNameBeanBus} 获取到处理器，当发生更新或删除操作时，会轮询这些处理器，串行处理。如果某个处理器发生异常，
 * 那么等待的处理器不会受到影响而继续执行。</p>
 *
 * @author allen
 * @since 1.0
 * @param <T>   数据模型
 * @param <Bus> 由多个 {@link ChangedHandler} 组成的 {@link AbstractNameBeanBus} 更新处理器工厂，这个工厂内部包含执行更新的处理器列表。
 * @see com.bxm.adx.common.caching.AbstractLoadingCache
 */
@Slf4j
public abstract class AbstractChangedHandlerLoadingCaching<T, Bus extends AbstractNameBeanBus<? extends ChangedHandler<T>>>
        extends AbstractLoadingCache<T> {

    /**
     * 返回更新处理器的工厂
     * @return 工厂
     */
    protected abstract AbstractNameBeanBus<? extends ChangedHandler<T>> getChangedHandlerFactory();

    @Override
    protected void afterUpdate(T old, T newest) {
        getChangedHandlerFactory().getAll().forEach(h -> {
            try {
                h.doUpdate(old, newest);
            } catch (Exception e) {
                log.error("doUpdate: ", e);
            }
        });
    }

    @Override
    protected void afterDelete(T old) {
        getChangedHandlerFactory().getAll().forEach(h -> {
            try {
                if (Objects.nonNull(old)) {
                    h.doDelete(old);
                }
            } catch (Exception e) {
                log.error("doDelete: ", e);
            }
        });
    }
}
