package com.bxm.adx.common.caching;

import com.bxm.adx.common.utils.ChangeUtils;
import com.bxm.adx.common.utils.ChangedEnum;

import java.util.function.Supplier;

/**
 * {@code T} 对象更新处理器
 *
 * @author allen
 * @date 2021-06-09
 * @since 1.0
 */
public interface ChangedHandler<T> {

    /**
     * 处理 T 对象更新逻辑
     * @param old 更新前
     * @param latest 更新后
     */
    void doUpdate(T old, T latest);

    /**
     * 处理 T 对象删除逻辑
     * @param old 删除前
     */
    void doDelete(T old);

    /**
     * 是否需要执行数据更新，并酌情删除更新前的某些数据
     * 用途：只判断整体数据是否新增数据<br/>
     *
     * @param old 更新前数据
     * @param latest 更新后数据
     * @return
     */
    default boolean isExecuteUpdateAndDoDelete(T old, T latest){
        ChangedEnum changedEnum = ChangeUtils.getChanged(old);
        if(ChangedEnum.ADD == changedEnum){
            // 新增整条数据
            return true;
        }else {
            doDelete(old);
            return true;
        }
    }

    /**
     * 是否需要执行数据更新，并酌情删除更新前的某些数据<br/>
     * 用途：详细判断某个节点数据是否有变更<br/>
     *
     * @param old 更新前数据
     * @param latest 更新后数据
     * @param oldNode 更新前数据某个节点
     * @param latestNode 更新后数据某个节点
     * @return
     */
    default boolean isExecuteUpdateAndDoDelete(T old, T latest, Supplier<Object> oldNode, Supplier<Object> latestNode){
        ChangedEnum changedEnum = ChangeUtils.getChanged(old);
        if(ChangedEnum.ADD == changedEnum){
            // 新增整条数据
            return true;
        }else {
            changedEnum = ChangeUtils.getNodeChanged(oldNode.get(), latestNode.get());
            if(ChangedEnum.NODE_NONE == changedEnum){
                // 该项信息无改动
                return false;
            }
            doDelete(old);
            return true;
        }
    }
}
