package com.bxm.adx.common.creative;

import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import lombok.extern.slf4j.Slf4j;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;

/**
 * For cache pushable
 *
 * <pre>
 *     name = "CREATIVE"
 *     parameters = {type=$type}
 *          type = 1.Creative 2.Ref
 *     byte[] = Creative or CreativePositionRef object to json bytes
 * </pre>
 * @author allen
 * @since 2019-12-18
 */
@Slf4j
@CachePush("CREATIVE")
public class CreativeDaoImpl implements CreativeDao, Pushable {

    private final Updater updater;
    private final Fetcher fetcher;

    public CreativeDaoImpl(Updater updater, Fetcher fetcher) {
        this.updater = updater;
        this.fetcher = fetcher;
    }

    @Override
    public Creative getCreative(Long id) {
        return fetcher.fetch(getCreativeKeyGenerator(id), Creative.class);
    }

    @Override
    public Creative getCreativeByDsp(Long dspId, String dspCreativeId) {
        return fetcher.hfetch(getCreativeOfDspKeyGenerator(dspId), dspCreativeId, Creative.class);
    }

    @Override
    public Collection<CreativePositionRef> getRefs(String positionId) {
        return fetcher.hfetchall(getCreativePositionRefKeyGenerator(positionId), CreativePositionRef.class).values();
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        Object o = parameters.get("type");
        if (Objects.isNull(o)) {
            return;
        }
        switch (TypeHelper.castToInt(o)) {
            case 1:
                this.updateCreative(JsonHelper.convert(data, Creative.class));
                break;
            case 2:
                this.updateCreativePositionRef(JsonHelper.convert(data, CreativePositionRef.class));
                break;
            default:
                log.warn("Unsupported type: {}", o);
                break;
        }
    }

    private void updateCreativePositionRef(CreativePositionRef ref) {
        String positionId = ref.getPositionId();
        Long creativeId = ref.getCreativeId();
        KeyGenerator keyGenerator = getCreativePositionRefKeyGenerator(positionId);
        updater.hupdate(keyGenerator, Objects.toString(creativeId), ref);
    }

    private KeyGenerator getCreativePositionRefKeyGenerator(String positionId) {
        return () -> KeyBuilder.build("ADX", "CREATIVE_POSITION_REF", "POSITIONID", positionId);
    }

    private void updateCreative(Creative creative) {
        updater.update(getCreativeKeyGenerator(creative.getId()), creative);
        updater.hupdate(getCreativeOfDspKeyGenerator(creative.getDspId()), creative.getDspCrtid(), creative);
    }

    private KeyGenerator getCreativeKeyGenerator(Long id) {
        return () -> KeyBuilder.build("ADX", "CREATIVE", "ID", id);
    }

    private KeyGenerator getCreativeOfDspKeyGenerator(Long dspId) {
        return () -> KeyBuilder.build("ADX", "CREATIVE", "DSP", dspId);
    }
}
