package com.bxm.adx.common.filter;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DotEventControlProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * DSP平均曝光限制过滤
 * @author fgf
 * @since 2022-11-21
 */
@Slf4j
@Component
public class DspAvgExposureLimiterFilter extends AbstractDspFilter {
    private final AdxProperties adxProperties;
    private final Fetcher fetcher;

    public DspAvgExposureLimiterFilter(AdxProperties adxProperties, Fetcher fetcher) {
        this.adxProperties = adxProperties;
        this.fetcher = fetcher;
    }

    @Override
    protected List<Dispatcher> doFilter(MarketOrders marketOrders) {
        List<MarketRequest> requests = marketOrders.getRequests();
        if (CollectionUtils.isEmpty(requests)) {
            return null;
        }
        LocalDateTime now = LocalDateTime.now();
        List<Dispatcher> list = Lists.newArrayList();

        requests.forEach(position -> {
            Set<BuyerWrapper> buyerWrappers = position.getBuyers();
            if (CollectionUtils.isEmpty(buyerWrappers)) {
                return;
            }
            buyerWrappers.forEach(buyerWrapper ->{
                Dispatcher dispatcher = buyerWrapper.getDispatcher();
                DotEventControlProperties control = needControl(dispatcher.getDspId().toString(), dispatcher.getDspPosid(), now);
                if (Objects.isNull(control)) {
                    return;
                }

                int count = getDspPositionExposureBySection(control, now);
                int allCount = getDspPositionExposureByControl(control);
                long limit = control.getIntervalLimit(allCount);
                
                if (count == 0) {
                    return;
                }

                if (count >= limit) {
                    list.add(dispatcher);
                }
            });
        });
        return list;
    }

    /**
     * 获取时间段已曝光次数
     *
     */
    private int getDspPositionExposureBySection(DotEventControlProperties control, LocalDateTime now) {
        String dspId = control.getDspId();
        String dspTagId = control.getDspPositionId();

        KeyGenerator keyGenerator = AdxKeyGenerator.Counter.getDspPositionExposureBySection(control.getId(), control.getSectionId(now),
                dspId, dspTagId);
        Long exposure = fetcher.fetch(keyGenerator, Long.class);
        return Objects.isNull(exposure) ? 0 : exposure.intValue();
    }

    /**
     * 获取控制配置id下已曝光次数
     * @param control
     * @return
     */
    private int getDspPositionExposureByControl(DotEventControlProperties control) {
        String dspId = control.getDspId();
        String dspTagId = control.getDspPositionId();

        KeyGenerator keyGenerator = AdxKeyGenerator.Counter.getDspPositionExposureByControl(control.getId(),
                dspId, dspTagId);
        
        Long exposure = fetcher.fetch(keyGenerator, Long.class);
        return Objects.isNull(exposure) ? 0 : exposure.intValue();
    }

    private DotEventControlProperties needControl(String dspId, String dspTagId, LocalDateTime now) {
        List<DotEventControlProperties> properties = adxProperties.getDotEventControls();
        if (CollectionUtils.isEmpty(properties)) {
            return null;
        }

        Optional<DotEventControlProperties> optional = properties.stream()
                .filter(p -> Objects.nonNull(p.getMt()) && OpenlogConstants.Mt.IMP_MT == p.getMt())
                .filter(p -> p.getDspId().equals(dspId))
                .filter(p -> p.getDspPositionId().equals(dspTagId))
                .filter(p -> now.isAfter(p.getStart()) && now.isBefore(p.getEnd()))
                .findFirst();
        return optional.isPresent() ?  optional.get() : null;
    }

    @Override
    public int order() {
        return 5;
    }
}
