package com.bxm.adx.common.filter;

import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * DSP最大曝光限制过滤
 * @author zhengwangeng
 * @since 2022-6-13
 */
@Slf4j
@Component
public class DspMaxExposureLimiterFilter extends AbstractDspFilter {

    private final Fetcher fetcher;

    public DspMaxExposureLimiterFilter(Fetcher fetcher) {
        this.fetcher = fetcher;
    }

    @Override
    protected List<Dispatcher> doFilter(MarketOrders marketOrders) {
        List<MarketRequest> requests = marketOrders.getRequests();
        if (CollectionUtils.isEmpty(requests)) {
            return null;
        }

        List<Dispatcher> list = Lists.newArrayList();

        requests.forEach(position -> {
            Set<BuyerWrapper> buyerWrappers = position.getBuyers();
            if (CollectionUtils.isEmpty(buyerWrappers)) {
                return;
            }
            buyerWrappers.forEach(buyerWrapper ->{
                Dispatcher dispatcher = buyerWrapper.getDispatcher();
                if (Objects.isNull(dispatcher.getMaxExposure())) {
                    return;
                }
                if (getPositionDspExposure(dispatcher) >= dispatcher.getMaxExposure()) {
                    list.add(dispatcher);
                }
            });

        });
        return list;
    }

    /**
     * 获取已曝光次数
     *
     * @param dispatcher
     * @return
     */
    private int getPositionDspExposure(Dispatcher dispatcher) {
        String configId = dispatcher.getConfigId().toString();
        String dspId = dispatcher.getDspId().toString();
        KeyGenerator keyGenerator = AdxKeyGenerator.Counter.getPositionDspExposure(configId, dspId);
        Long exposure = fetcher.fetch(keyGenerator, Long.class);
        return Objects.isNull(exposure) ? 0 : exposure.intValue();
    }

    @Override
    public int order() {
        return 5;
    }
}
