package com.bxm.adx.common.log.dsplog;


import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.distributed.DisableDistributed;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 记录dsp的请求和响应 分开文件进行记录
 *
 * @author zhangdong
 * @date 2022/11/30
 */
@Component
@Slf4j
public class DspLogRecord implements DisposableBean {

    private final ObjectMapper mapper = new ObjectMapper();

    private final Logger requestLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".request");

    private final Logger responseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".response");

    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(10, 20, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(1000), new NamedThreadFactory("dspLogResponse"), new ThreadPoolExecutor.CallerRunsPolicy());

    private final ConcurrentHashMap<String, Long> map = new ConcurrentHashMap<>();


    @PostConstruct
    public void init() {
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    public void dspLog(BidRequest request, BidResponse response, Dispatcher dispatcher) {
        try {
            logRequest(request);
            //填充数据
            if (StringUtils.isEmpty(response.getId())) {
                response.setId(request.getId());
            }
            for (SeatBid seatBid : response.getSeat_bid()) {
                seatBid.setDspId(dispatcher.getDspId());
            }
            logResponse(response);
        } catch (Exception e) {}
    }


    private void logResponse(BidResponse obj) {
        if (obj == null) {
            return;
        }
        try {
            String s = mapper.writeValueAsString(obj);
            executor.execute(() -> responseLogger.info(s));
        } catch (Exception e) {
            log.error("jackson json error", e);
        }

    }


    private void logRequest(BidRequest obj) {
        if (obj == null) {
            return;
        }
        if (map.containsKey(obj.getId())) {
            return;
        }
        map.put(obj.getId(), System.currentTimeMillis());
        try {
            String s = mapper.writeValueAsString(obj);
            executor.execute(() -> requestLogger.info(s));
        } catch (Exception e) {
            log.error("jackson json error", e);
        }

    }

    @DisableDistributed
    @Scheduled(cron = "*/1 * * * * ?")
    public void deleteExpireKey() {
        long l = System.currentTimeMillis();
        for (Map.Entry<String, Long> entry : map.entrySet()) {
            if ((l - entry.getValue()) > 1000) {
                map.remove(entry.getKey());
            }
        }
    }


    @Override
    public void destroy() {
        executor.shutdown();
        try {
            executor.awaitTermination(5, TimeUnit.SECONDS);
        } catch (Exception ignored) {

        }
    }
}
