package com.bxm.adx.common.market;

import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import org.apache.commons.lang.RandomStringUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 成交信息，包含买家、卖家等信息
 *
 * @author allen
 * @since 2019-12-13
 */
public class Deal {
    public final static byte HTTP_DEAL = 0;
    public final static byte CACHE_DEAL = 1;
    private byte type = HTTP_DEAL;
    private Buyer buyer;
    private Dispatcher dispatcher;
    private byte[] request;
    private byte[] response;
    private BidRequest bidRequest;
    private BidRequest adxRequest;
    private BidResponse bidResponse;
    private AdxErrEnum adxErrEnum;
    private Boolean bidSuccess;

    public Deal(Dispatcher dispatcher, BidRequest bidRequest) {
        this.dispatcher = dispatcher;
        this.bidRequest = bidRequest;
    }

    public Deal(Buyer buyer, BidRequest bidRequest, BidResponse bidResponse, Dispatcher dispatcher) {
        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.bidResponse = bidResponse;
        this.dispatcher = dispatcher;
    }

    public Deal(byte type, Buyer buyer, BidRequest bidRequest, BidResponse bidResponse, Dispatcher dispatcher) {
        this.type = type;
        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.bidResponse = bidResponse;
        this.dispatcher = dispatcher;
        if (type == CACHE_DEAL) {
            this.bidSuccess = true;
        }
    }

    public Deal(Buyer buyer, byte[] request, byte[] response, BidRequest bidRequest) {
        this.buyer = buyer;
        this.request = request;
        this.response = response;
        this.bidRequest = bidRequest;
    }

    public Deal(Buyer buyer, byte[] request, byte[] response, BidRequest bidRequest, BidRequest adxRequest, Dispatcher dispatcher) {
        this.buyer = buyer;
        this.request = request;
        this.response = response;
        this.bidRequest = bidRequest;
        this.adxRequest = adxRequest;
        this.dispatcher = dispatcher;
    }

    public Deal(Buyer buyer, byte[] request, BidRequest bidRequest, BidRequest adxRequest, AdxErrEnum adxErrEnum, Dispatcher dispatcher) {
        this.buyer = buyer;
        this.request = request;
        this.bidRequest = bidRequest;
        this.adxRequest = adxRequest;
        this.adxErrEnum = adxErrEnum;
        this.dispatcher = dispatcher;
    }

    /**
     * @return 是否竞价成功，有广告返回
     */
    public boolean isBidSuccess() {
        if (Objects.isNull(this.bidSuccess)) {
            this.bidSuccess = Objects.nonNull(getBidResponse()) && hasBid();
        }
        return bidSuccess;
    }

    public BidResponse getBidResponse() {
        if (null == bidResponse) {
            BuyModelAdapter modelAdapter = buyer.getModelAdapter();
            if (null != response && response.length > 0) {
                this.bidResponse = modelAdapter.buildResponse(response);
            }
        }
        return bidResponse;
    }

    public boolean hasBid() {
        if (Objects.nonNull(bidResponse.getNbr())) {
            return false;
        }

        List<SeatBid> seatBids = bidResponse.getSeat_bid();
        if (CollectionUtils.isEmpty(seatBids)) {
            return false;
        }
        for (SeatBid seatBid : seatBids) {
            List<Bid> bids = seatBid.getBid();
            if (!CollectionUtils.isEmpty(bids)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断是否是广告优化dsp，广告优化dsp不需要返回正常广告，只需要用到其返回的曝光链接
     * @return
     */
    public boolean isOp() {
        String code = buyer.getCode();
        if ("optimization".equalsIgnoreCase(code)) {
            return true;
        }
        return false;
    }

    public Buyer getBuyer() {
        return buyer;
    }

    public void setBuyer(Buyer buyer) {
        this.buyer = buyer;
    }

    public byte[] getRequest() {
        return request;
    }

    public void setRequest(byte[] request) {
        this.request = request;
    }

    public byte[] getResponse() {
        return response;
    }

    public void setResponse(byte[] response) {
        this.response = response;
    }

    public BidRequest getBidRequest() {
        return bidRequest;
    }

    public AdxErrEnum getAdxErrEnum() {
        return adxErrEnum;
    }

    public BidRequest getAdxRequest() {
        return adxRequest;
    }

    public Dispatcher getDispatcher() {
        return dispatcher;
    }

    public byte getType() {
        return type;
    }

    public Boolean getBidSuccess() {
        return bidSuccess;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Deal deal = (Deal) o;
        return getType() == deal.getType() && getDispatcher().equals(deal.getDispatcher());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getType(), getDispatcher());
    }
}
