package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DspThreadPoolProperties;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.log.dsplog.DspLogRecord;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.exchange.rebuild.request.AdxBidRequestBuildFactory;
import com.bxm.adx.common.market.exchange.rebuild.request.RequestBuildAttribute;
import com.bxm.adx.common.market.exchange.rebuild.response.AdxBidResponseBuildFactory;
import com.bxm.adx.common.market.filter.DealFilterFactory;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.micrometer.ExchangeCallableMeter;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.rebuild.request.RebuildRequestChain;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.*;

/**
 * @author allen
 * @since 2019-12-12
 */
@Slf4j
@Configuration
public class RtbExchanger implements Exchanger, DisposableBean {

    private final BuyerMeter buyerMeter;
    private final ExchangeCallableMeter exchangeCallableMeter;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final DataLogDao dataLogDao;
    private final ThreadPoolExecutor executor;
    private final DspLogRecord dspLogRecord;
    private final RebuildRequestChain requestChain;
    private final AdxBidRequestBuildFactory factory;
    private final AdxBidResponseBuildFactory responseBuildFactory;
    private final DealFilterFactory dealFilterFactory;
    public RtbExchanger(BuyerMeter buyerMeter, ExchangeCallableMeter exchangeCallableMeter, EventPark eventPark,
                        BuyerResponseCache buyerResponseCache, DataLogDao dataLogDao,
                        CustomRejected customRejected, AdxProperties adxProperties, DspLogRecord dspLogRecord,
                        RebuildRequestChain requestChain, AdxBidRequestBuildFactory factory,
                        AdxBidResponseBuildFactory responseBuildFactory, DealFilterFactory dealFilterFactory) {
        this.buyerMeter = buyerMeter;
        this.exchangeCallableMeter = exchangeCallableMeter;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.dataLogDao = dataLogDao;
        this.dspLogRecord = dspLogRecord;
        this.requestChain = requestChain;
        this.factory = factory;
        this.responseBuildFactory = responseBuildFactory;
        this.dealFilterFactory = dealFilterFactory;
        DspThreadPoolProperties dspThreadPool = adxProperties.getDspThreadPool();
        this.executor = new ThreadPoolExecutor(dspThreadPool.getCoreCount(), dspThreadPool.getMaxCount(), dspThreadPool.getLiveTime(), TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(dspThreadPool.getQueueLength()), new NamedThreadFactory("exchange"), customRejected);
    }

    @Override
    public ExchangeResult bidding(BidRequest bidRequest, Collection<BuyerWrapper> buyers) {
        ExchangeResult result = new ExchangeResult();
        result.getBuyers().addAll(buyers);
        if (0 == buyers.size() || null == bidRequest) {
            return result;
        }
        List<Future<Deal>> futures = Lists.newArrayList();
        CountDownLatch waitCountDown = new CountDownLatch(buyers.size());
        CountDownLatch overtimeCountDown = new CountDownLatch(1);
        //遍历提交到线程池
        for (final BuyerWrapper buyer : buyers) {
            //发送http请求
            BidRequest adxBidRequest = factory.buildAdxBidRequest(
                    bidRequest,
                    RequestBuildAttribute.builder()
                            .buyerWrapper(buyer)
                            .build()
            );
            futures.add(executor.submit(new ExchangeCallable(buyer, bidRequest, adxBidRequest, buyerMeter, eventPark,
                    buyerResponseCache, waitCountDown, overtimeCountDown, dataLogDao, dspLogRecord, requestChain,
                    responseBuildFactory, exchangeCallableMeter)));
            //查询缓存数据
            Dsp dsp = buyer.getBuyer().getDsp();
            Long cacheTime = dsp.getCacheTime();
            if (Objects.isNull(cacheTime) || cacheTime.intValue() == 0) {
                continue;
            }
            futures.add(executor.submit(new ExchangeCacheCallable(bidRequest, buyer, buyerResponseCache, exchangeCallableMeter)));
        }
        //正常处理
        waitDealFilter(result, waitCountDown, bidRequest.getWaitTime(), futures);
        if (!CollectionUtils.isEmpty(result.getDeals())) {
            return result;
        }
        if (futures.size() == 0) {
            return result;
        }
        //超时处理
        waitDealFilter(result, overtimeCountDown, bidRequest.getOvertime() - bidRequest.getWaitTime(), futures);
        return result;
    }

    private void waitDealFilter(ExchangeResult result, CountDownLatch waitCountDown, Long waitTime, List<Future<Deal>> dealList) {
        try {
            waitCountDown.await(waitTime, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            log.error("countDown err", e);
            return;
        }
        //获取结果
        List<Deal> deals = Lists.newArrayList();
        Iterator<Future<Deal>> iterator = dealList.iterator();
        while (iterator.hasNext()) {
            Future<Deal> future = iterator.next();
            try {
                if (future.isDone()) {
                    Deal response = future.get();
                    if (null != response) {
                        deals.add(response);
                    }
                    iterator.remove();
                }
            } catch (InterruptedException | ExecutionException e) {
                // 超时或意外终止
                future.cancel(true);
                log.error("execute: ", e);
            }
        }
        //前置过滤
        Map<Integer, Set<Deal>> trashMap = dealFilterFactory.filterDeals(deals, true);
        result.getDeals().addAll(deals);
        result.addTrash(trashMap);
        return;
    }

    @Override
    public void destroy() {
        executor.shutdown();
    }
}
