package com.bxm.adx.common.market.exchange.rebuild.offer;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

/**
 * @author fgf
 * @date 2023/1/10
 **/
@Configuration
public class NoneDspAvgPriceOffer implements Offer {
    @Override
    public Map<Bid, OfferResult> offer(Collection<Bid> bids, BidRequest request, Dispatcher dispatcher) {
        Map<Bid, OfferResult> offerResultMap = new HashMap<>(bids.size());
        for (Bid bid : bids) {
            BigDecimal offer = getPriceByConfig(bid, request, dispatcher);
            offerResultMap.put(bid, OfferResult.builder()
                            .offer(offer)
                            .chargeType(AdxConstants.ChargeType.CPM)
                    .build());
        }
        return offerResultMap;
    }

    @Override
    public OfferType offerType() {
        return OfferType.NONE_Dsp_Avg_Price;
    }

    /**
     * 获取配置出价配置的情况下的出价
     * @param bidRequest
     * @param dispatcher
     * @return
     */
    private BigDecimal getPriceByConfig(Bid bid, BidRequest bidRequest, Dispatcher dispatcher) {
        DispatcherPriceConfig config = dispatcher.getDispatcherPriceConfig();
        if (Objects.isNull(config)) {
            return bid.getPrice();
        }
        //媒体出价=dsp出价*（1-利润率）*出价系数 注:此处dsp出价是经过流量分配中广告主系数处理后的出价，非原始dsp出价
        BigDecimal dspWinPrice = bid.getDspWinPrice();
        BigDecimal profitMargin = Optional.ofNullable(config.getProfitMargin()).orElse(BigDecimal.ZERO);
        BigDecimal newPrice = dspWinPrice
                .multiply(new BigDecimal(100.00).subtract(profitMargin))
                .multiply(config.getBiddingCoefficient()).movePointLeft(2);
//        BigDecimal maxBidFloor = maxBidFloor(bidRequest, dispatcher);
        BigDecimal bidFloor = bidRequest.getImps().iterator().next().getBid_floor();
        if (newPrice.compareTo(bidFloor) >= 0) {
            return newPrice.setScale(0, RoundingMode.HALF_UP);
        } else {
            //探索出价-底价加1分钱
            BigDecimal explorePrice = bidFloor.add(BigDecimal.ONE);
            if (dspWinPrice.compareTo(explorePrice) >= 0) {
                return explorePrice.setScale(0, RoundingMode.HALF_UP);
            }
        }

        return null;
    }

    /**
     * since 20230522 逻辑修改，不取两者最大值，直接和原始媒体底价做比较
     *
     * 取配置底价和请求底价中较大的那个
     * @param bidRequest
     * @param dispatcher
     * @return
     */
    @Deprecated
    private BigDecimal maxBidFloor(BidRequest bidRequest, Dispatcher dispatcher) {
        //元转分
        BigDecimal dispatcherBidFloor = Optional.ofNullable(dispatcher.getDspBasePrice()).orElse(BigDecimal.ZERO).movePointRight(2);
        BigDecimal bidFloor = bidRequest.getImps().iterator().next().getBid_floor();
        int result = bidFloor.compareTo(dispatcherBidFloor);
        switch (result) {
            case -1:
                return dispatcherBidFloor;
            case 0:
            case 1:
            default:
                return bidFloor;
        }
    }
}
