package com.bxm.adx.common.market.exchange.rebuild.request;

import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.position.AdvertPoint;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.buy.position.Constants;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.common.sell.request.Native;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * * 重构BidRequest中的Impression
 *
 * @author fgf
 * @date 2023/1/6
 **/
@Configuration
public class ImpressionBuilder implements AdxBidRequestBuilder {
    private final AdvertPointService advertPointService;

    public ImpressionBuilder(AdvertPointService advertPointService) {
        this.advertPointService = advertPointService;
    }

    @Override
    public void rebuildAdxBidRequest(BidRequest adxBidRequest, RequestBuildAttribute attribute) {
        Buyer buyer = attribute.getBuyerWrapper().getBuyer();
        Dispatcher dispatcher = attribute.getBuyerWrapper().getDispatcher();
        AdvertPoint advertPoint = advertPointService.getAdvertPointByDspId(dispatcher.getDspId().toString(),
                dispatcher.getDspAppid(), dispatcher.getDspPosid());

        List<Impression> oldImpressions = adxBidRequest.getImps();
        List<Impression> newImpressions = new ArrayList<>();
        for (Impression impression : oldImpressions) {
            //init newImpression
            Impression newImpression = new Impression();
            BeanUtils.copyProperties(impression, newImpression);

            //替换DSP广告位ID
            if (needReplaceTagIdByDsp(adxBidRequest, buyer)) {
                newImpression.setTag_id(dispatcher.getDspPosid());
            }
            //根据Dsp-广告端点替换
            if (needReplaceByAdvertPoint(adxBidRequest)) {
                handleAdvertPoint(advertPoint, newImpression, buyer);
            }
            //底价处理
            handleBidFloor(newImpression, dispatcher);
            newImpressions.add(newImpression);
            break;
        }
        adxBidRequest.setImps(newImpressions);
    }

    /**
     * 处理Dsp-广告位端点
     *
     * @param advertPoint
     * @param newImpression
     * @param buyer
     */
    private void handleAdvertPoint(AdvertPoint advertPoint, Impression newImpression, Buyer buyer) {
        if (Objects.nonNull(advertPoint)) {
            //替换广告位尺寸
            AdvertPoint.Size size = advertPoint.getImpressionSize();
            if (Objects.nonNull(size)) {
                newImpression.setH(size.getH());
                newImpression.setW(size.getW());
            }
            //素材模版id
            if (StringUtils.isNotEmpty(advertPoint.getTemplateId())) {
                Native an = new Native();
                if (Objects.nonNull(newImpression.getA_native())) {
                    BeanUtils.copyProperties(newImpression.getA_native(), an);
                }
                an.setTemplate(advertPoint.getTemplateId());
                newImpression.setA_native(an);
            }
            //启航处理广告位类型
            Integer positionType = advertPoint.getPositionType();
            if (Objects.nonNull(positionType)) {
                if ("qihang".equalsIgnoreCase(buyer.getCode())) {
                    Integer newImpType = Constants.getPositionSceneTypeByPositionType(positionType);
                    if (Objects.nonNull(newImpType)) {
                        newImpression.setImp_type(newImpType);
                    }
                }
            }
        }
    }

    /**
     * 底价处理
     *
     * @param newImpression
     * @param dispatcher
     */
    private void handleBidFloor(Impression newImpression, Dispatcher dispatcher) {
        //优先使用流量分配中的底价
        BigDecimal dspBasePrice = dispatcher.getDspBasePrice();
        if (Objects.nonNull(dspBasePrice)) {
            newImpression.setBid_floor(dspBasePrice.movePointRight(2));
        } else {
            //底价系数处理
            BigDecimal coefficient = dispatcher.getBasePriceCoefficient();
            if (Objects.nonNull(coefficient) && coefficient.compareTo(BigDecimal.ZERO) > 0) {
                newImpression.setBid_floor(newImpression.getBid_floor().divide(coefficient, 0, BigDecimal.ROUND_UP));
            }
        }

        //底价单位为分，不保留小数，有小数自动向上进位
        newImpression.setBid_floor(newImpression.getBid_floor().setScale(0, BigDecimal.ROUND_UP));
    }

    /**
     * 是否需要替换tagId
     *
     * @param request
     * @param buyer
     * @return
     */
    private boolean needReplaceTagIdByDsp(BidRequest request, Buyer buyer) {
        DockingMethodTypeEnum typeEnum = dockingMethodTypeEnum(request);
        if (Objects.isNull(typeEnum)) {
            return true;
        }
        switch (typeEnum) {
            case SDK_OPERATION:
                if ("scene".equalsIgnoreCase(buyer.getCode())) {
                    return false;
                }
                return true;
            default:
                return true;
        }
    }

    /**
     * 是否需要根据Dsp-广告端点信息替换Impression
     *
     * @param request
     * @return
     */
    private boolean needReplaceByAdvertPoint(BidRequest request) {
        DockingMethodTypeEnum typeEnum = dockingMethodTypeEnum(request);
        if (DockingMethodTypeEnum.SDK_OPERATION.equals(typeEnum)) {
            return false;
        }
        return true;
    }

    /**
     * 广告位对接类型
     *
     * @param request
     * @return
     */
    private DockingMethodTypeEnum dockingMethodTypeEnum(BidRequest request) {
        Integer dockingMethodType = request.getDockingMethodType();
        DockingMethodTypeEnum typeEnum = DockingMethodTypeEnum.get(dockingMethodType);
        return typeEnum;
    }
}
