package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AbstractPluginBuyModelAdapter;
import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.ImpMonitor;
import com.bxm.adx.common.sell.response.SeatBid;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author fgf
 * @date 2023/5/15
 **/
@Configuration
@Slf4j
public class DspWinPriceBuilder implements AdxBidResponseBuilder {
    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        Buyer buyer = attribute.getBuyer();
        Dsp dsp = buyer.getDsp();
        String dspKey = dsp.getApiKey();
        BuyModelAdapter adapter = buyer.getModelAdapter();
        if (!(adapter instanceof AbstractPluginBuyModelAdapter)) {
            return;
        }
        AbstractPluginBuyModelAdapter pluginAdapter = (AbstractPluginBuyModelAdapter) adapter;
        String macro = pluginAdapter.getPluginConfig().getProperties().getProperty(AdxConstants.PluginParam.MACRO_WIN_PRICE);
        if (StringUtils.isBlank(macro)) {
            return;
        }
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                if (Objects.nonNull(bid.getDspWinPrice())) {
                    continue;
                }
                BigDecimal price = bid.getDsp_price();
                BigDecimal dspWinPrice = getWinPrice(price, attribute.getDispatcher());
                dspWinPrice = pluginAdapter.priceScale(dspWinPrice);
                bid.setDspWinPrice(dspWinPrice);

                String cipher = pluginAdapter.encrypt(dspWinPrice.toString(), dspKey);
                if (StringUtils.isBlank(cipher)) {
                    continue;
                }
                List<ImpMonitor> impMonitors = bid.getImp_monitors();
                if (CollectionUtils.isEmpty(impMonitors)) {
                    continue;
                }
                impMonitors.forEach(
                        impMonitor -> {
                            String imp = impMonitor.getImp_monitor_url().replace(macro, cipher);
                            impMonitor.setImp_monitor_url(imp);
                        }
                );
            }
        }
    }

    @Override
    public int getOrder() {
        return 3;
    }

    /**
     * 赢价计算
     * 例子：广告主出价10元，8元以内部分折扣系数0.8，8元以外部分0.4，则上报广告主赢价=8*0.8+（10-8）*0.4
     * @param dspOffer
     * @return
     */
    private static BigDecimal getWinPrice(BigDecimal dspOffer, Dispatcher dispatcher) {
        DispatcherPriceConfig config = dispatcher.getDispatcherPriceConfig();
        //未配置出价配置直接返回原出价
        if (Objects.isNull(config)) {
            return dspOffer;
        }
        BigDecimal priceSplit = config.getPriceSplit();
        BigDecimal lt = config.getLtPriceDiscountFactor();
        BigDecimal gt = config.getGtPriceDiscountFactor();
        //未配置的时段使用默认系数
        if (Objects.isNull(priceSplit)) {
            lt = Optional.ofNullable(lt).orElse(BigDecimal.ONE);
            return dspOffer.multiply(lt);
        } else {
            priceSplit = priceSplit.movePointRight(2);
            //看注释例子
            int compare = dspOffer.compareTo(priceSplit);
            if (compare <= 0) {
                return dspOffer.multiply(lt);
            } else {
                BigDecimal ltPrice = priceSplit.multiply(lt);
                BigDecimal gtPrice = dspOffer.subtract(priceSplit).multiply(gt);
                return ltPrice.add(gtPrice);
            }
        }
    }
}
