package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.exchange.ExchangeResult;
import com.bxm.adx.common.openlog.event.internal.AdxFilterEvent;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @since 2019-12-18
 */
@Configuration
@Slf4j
public class DealFilterFactory implements ApplicationListener<ApplicationReadyEvent> {

    private Collection<Filter> filters;
    private final EventPark eventPark;

    public DealFilterFactory(EventPark eventPark) {
        this.eventPark = eventPark;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        this.filters = event.getApplicationContext().getBeansOfType(Filter.class).values();
        this.filters = this.filters.stream().sorted(Comparator.comparing(Filter::getOrder)).collect(Collectors.toList());
    }

    public void filter(MarketOrders orders, ExchangeResult result) {
        List<Deal> deals = result.getDeals();
        if (log.isDebugEnabled()) {
            log.debug("before filter deal list size={}, dispatcher={}", deals.size(),
                    deals.stream().map(Deal::getDispatcher).map(Dispatcher::getId).collect(Collectors.toSet()));
        }
        if (!CollectionUtils.isEmpty(deals)) {
            //过滤
            Map<Integer, Set<Deal>> trashMap = filterDeals(deals, false);
            //添加被过滤的信息（包括超时没返回的）
            result.addTrash(trashMap);
        }
        result.addTrash(getOvertimeDeal(result, orders.getBidRequest()));
        if (!CollectionUtils.isEmpty(result.getTrash())) {
            eventPark.post(new AdxFilterEvent(this, orders.getBidRequest(), result.getTrash()));
        }
    }

    /**
     * 过滤成交结果deal
     * @param deals
     * @param preview
     * @return
     */
    public Map<Integer, Set<Deal>> filterDeals(List<Deal> deals, boolean preview) {
        Set<Deal> trash = Sets.newHashSet();
        Map<Integer, Set<Deal>> trashMap = Maps.newHashMap();
        for (Filter filter : filters) {
            if (!checkPreview(preview, filter)) {
                continue;
            }
            String clazzSimpleName = ClassUtils.getUserClass(filter).getSimpleName();
            filter.filter(deals, trash);
            if (!CollectionUtils.isEmpty(trash)) {
                if (log.isDebugEnabled()) {
                    log.debug("filter {} filter dispatcher={}", clazzSimpleName,
                            trash.stream().map(Deal::getDispatcher).map(Dispatcher::getId).collect(Collectors.toSet()));
                }
                int filterType = Arrays.stream(FilterEnum.values())
                        .filter(filterEnum -> filterEnum.name().equals(clazzSimpleName))
                        .findFirst()
                        .orElse(FilterEnum.Undefined)
                        .getType();
                MapHelper.get(trashMap, filterType, new HashSet<>()).addAll(trash);
                deals.removeIf(
                        deal -> trash.contains(deal)
                );
                trash.clear();
            }
            if (CollectionUtils.isEmpty(deals)) {
                break;
            }
        }
        return trashMap;
    }

    /**
     * 检查是否是前置过滤器
     * 前置过滤器在rtbExchange中起效
     * @param preview
     * @param filter
     * @return
     */
    private boolean checkPreview(boolean preview, Filter filter) {
        DealFilterCondition condition = AnnotationUtils.findAnnotation(ClassUtils.getUserClass(filter), DealFilterCondition.class);
        boolean conPreview = false;
        if (Objects.nonNull(condition)) {
            conPreview = condition.preview();
        }
        if (preview) {
            if (conPreview) {
                return true;
            } else {
                return false;
            }
        } else {
            if (conPreview) {
                return false;
            } else {
                return true;
            }
        }
    }

    /**
     * 根据已请求dispatcher减去返回deal的dispatcher计算超时的dispatcher
     *
     * @param exchangeResult
     * @param bidRequest
     * @return
     */
    private Map<Integer, Set<Deal>> getOvertimeDeal(ExchangeResult exchangeResult, BidRequest bidRequest) {
        Collection<BuyerWrapper> requested = exchangeResult.getBuyers();
        if (CollectionUtils.isEmpty(requested)) {
            return null;
        }
        Set<Dispatcher> requestedDispatcher = requested.stream().map(BuyerWrapper::getDispatcher).collect(Collectors.toSet());
        Set<Dispatcher> dealDispatcher = exchangeResult.getDeals().stream().map(Deal::getDispatcher).collect(Collectors.toSet());
        if (!CollectionUtils.isEmpty(exchangeResult.getTrash())) {
            dealDispatcher.addAll(
                    exchangeResult.getTrash().values().stream().flatMap(Collection::stream).map(Deal::getDispatcher).collect(Collectors.toSet())
            );
        }
        requestedDispatcher.removeIf(
                dispatcher -> dealDispatcher.contains(dispatcher)
        );

        if (CollectionUtils.isEmpty(requestedDispatcher)) {
            return null;
        }
        Set<Deal> overtime = new HashSet<>(requestedDispatcher.size());
        for (Dispatcher dispatcher : requestedDispatcher) {
            overtime.add(new Deal(dispatcher, bidRequest));
        }
        Map<Integer, Set<Deal>> trash = Maps.newHashMapWithExpectedSize(1);
        trash.put(FilterEnum.Overtime.getType(), overtime);
        return trash;
    }
}
