package com.bxm.adx.common.micrometer.aspect;

import com.bxm.adx.common.market.exchange.Exchanger;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author fgf
 * @date 2023/3/7
 **/
@Slf4j
@Aspect
public class ExchangeMicroMeterAspect {
    private final ExchangeMicroMeter microMeter;

    public ExchangeMicroMeterAspect(ExchangeMicroMeter microMeter) {
        this.microMeter = microMeter;
    }

    @Pointcut("this(com.bxm.adx.common.market.exchange.Exchanger) && execution(* bidding(..))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Object target = point.getTarget();
        if (!(target instanceof Exchanger)) {
            return point.proceed();
        }

        Exchanger exchanger = (Exchanger) target;
        counterBuyer(point.getArgs(), exchanger);
        Timer timer = microMeter.getTimer(exchanger);
        long start = System.nanoTime();
        try {
            return point.proceed();
        } finally {
            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
    }

    private void counterBuyer(Object[] args, Exchanger exchanger) {
        if (ArrayUtils.isEmpty(args)) {
            return;
        }

        Object arg = args[1];
        if (Objects.isNull(arg)) {
            return;
        }

        if (arg instanceof Collection) {
            Collection collection = (Collection) arg;
            microMeter.getCounter(exchanger).increment(collection.size());
        }
    }
}
