package com.bxm.adx.common.report;

import com.bxm.openlog.extension.client.HttpClientUtils;
import com.bxm.warcar.utils.NamedThreadFactory;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;
import org.springframework.util.ClassUtils;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * @author fgf
 * @date 2023/4/19
 **/
@Slf4j
public class DefaultHttpClientReportClient implements ReportClient {
    private final ThreadPoolExecutor executor;
    private final HttpClient httpClient;

    private Timer timer;
    private Counter success;
    private Counter fail;

    private DefaultHttpClientReportClient(ThreadPoolExecutor executor, int maxTotal, int defaultMaxPerRoute, int connectionRequestTimeout, int connectTimeout, int socketTimeout) {
        this.executor = executor;
        this.httpClient = HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
    }

    public static DefaultHttpClientReportClient.Builder builder() {
        return new DefaultHttpClientReportClient.Builder();
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        Class<?> clazz = ClassUtils.getUserClass(this);
        registerGauge(registry, this, clazz);
        registerTimer(registry, this, clazz);
        registerCounter(registry, this, clazz);
    }

    @Override
    public boolean request(String url, Header... headers) throws IOException {
        long start = System.nanoTime();
        HttpResponse response = null;
        try {
            HttpGet get = new HttpGet(url);
            if (ArrayUtils.isNotEmpty(headers)) {
                get.setHeaders(headers);
            }
            response = httpClient.execute(get);
            return response.getStatusLine().getStatusCode() == HttpStatus.SC_OK;
        } finally {
            if (Objects.nonNull(response)) {
                EntityUtils.consume(response.getEntity());
            }

            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
    }

    @Override
    public void asyncRequest(String url, Consumer<ReportFallback> fallback, Header... headers) {
        this.executor.execute(() -> {
            try {
                if (!request(url, headers)) {
                    fail.increment();
                    if (Objects.nonNull(fallback)) {
                        fallback.accept(ReportFallback.builder().url(url).headers(headers).build());
                    }
                } else {
                    success.increment();
                }
            } catch (IOException e) {
                fail.increment();

                if (Objects.nonNull(fallback)) {
                    fallback.accept(ReportFallback.builder().url(url).headers(headers).exception(e).build());
                }
            }
        });
    }

    @Override
    public void asyncRequest(String url, Header... headers) {
        this.asyncRequest(url, null, null);
    }

    @Override
    public int queueSize() {
        return executor.getQueue().size();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(30, TimeUnit.SECONDS)) {
                log.warn("This executor was forced terminated!");
            }
        } catch (InterruptedException e) {
            log.error("awaitTermination: ", e);
        }
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public ThreadPoolExecutor getExecutor() {
        return executor;
    }

    private void registerGauge(MeterRegistry registry, ReportClient client, Class<?> clazz) {
        Gauge.builder("ReportClient.queue", 0, value -> client.queueSize())
                .tags("name", clazz.getName())
                .register(registry);
    }

    private void registerTimer(MeterRegistry registry, ReportClient client, Class<?> clazz) {
        this.timer = Timer.builder("ReportClient.timer")
                .tag("name", clazz.getName())
                .register(registry);
        log.info("Registering ReportClient {} timer meter successful.", client);
    }

    private void registerCounter(MeterRegistry registry, ReportClient client, Class<?> clazz) {
        this.success = Counter.builder("ReportClient.counter")
                .tag("name", "success")
                .register(registry);
        this.fail = Counter.builder("ReportClient.counter")
                .tag("name", "fail")
                .register(registry);
        log.info("Registering ReportClient {} counter meter successful.", client);
    }

    public static class Builder {

        private int maxTotal = 200;
        private int defaultMaxPerRoute = 20;
        private int connectionRequestTimeout = 100;
        private int connectTimeout = 200;
        private int socketTimeout = 500;
        private ThreadPoolExecutor executor;

        {
            int processors = Runtime.getRuntime().availableProcessors();
            int max = 10000;
            executor = new ThreadPoolExecutor(processors, processors, 0, TimeUnit.SECONDS,
                    new LinkedBlockingQueue<>(max), new NamedThreadFactory("report-client"));
        }

        private Builder() {}

        public DefaultHttpClientReportClient build() {
            return new DefaultHttpClientReportClient(executor, maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
        }

        public DefaultHttpClientReportClient.Builder setMaxTotal(int maxTotal) {
            this.maxTotal = maxTotal;
            return this;
        }

        public DefaultHttpClientReportClient.Builder setDefaultMaxPerRoute(int defaultMaxPerRoute) {
            this.defaultMaxPerRoute = defaultMaxPerRoute;
            return this;
        }

        public DefaultHttpClientReportClient.Builder setConnectionRequestTimeout(int connectionRequestTimeout) {
            this.connectionRequestTimeout = connectionRequestTimeout;
            return this;
        }

        public DefaultHttpClientReportClient.Builder setConnectTimeout(int connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public DefaultHttpClientReportClient.Builder setSocketTimeout(int socketTimeout) {
            this.socketTimeout = socketTimeout;
            return this;
        }

        public DefaultHttpClientReportClient.Builder setExecutor(ThreadPoolExecutor executor) {
            this.executor = executor;
            return this;
        }
    }
}
