package com.bxm.adx.common.report;

import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.ServerReportProperties;
import com.bxm.warcar.cache.KeyGenerator;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.message.BasicHeader;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;

import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * @author fgf
 * @date 2023/4/19
 **/
@Slf4j
public class ServerReportService {
    private final JedisPool jedisPool;
    private final ReportClient reportClient;
    private final ServerReportProperties serverReportProperties;

    public ServerReportService(ServerReportProperties serverReportProperties, JedisPool jedisPool, ReportClient reportClient) {
        this.serverReportProperties = serverReportProperties;
        this.jedisPool = jedisPool;
        this.reportClient = reportClient;
    }

    public void report(ReportType type, String bidId, MultiValueMap<String, String> headers) {
        String key = getKeyGenerator(type, bidId).generateKey();
        BasicHeader[] basicHeaders = convertToBasicHeaders(headers);
        for (; ; ) {
            try (Jedis jedis = jedisPool.getResource()) {
                jedis.select(serverReportProperties.getQueueDb());
                String url = jedis.lpop(key);
                if (StringUtils.isEmpty(url)) {
                    break;
                }
                reportClient.asyncRequest(url, new Consumer<ReportFallback>() {
                    @Override
                    public void accept(ReportFallback reportFallback) {
                        Exception exception = reportFallback.getException();
                        log.warn("report fail {} url {}, headers {}",
                                exception == null ? "" : exception.getMessage(), reportFallback.getUrl(), reportFallback.getHeaders());
                    }
                }, basicHeaders);
            } catch (Exception e) {
                log.error("report err {}, affect {} {} {}", e.getMessage(), type.name(), bidId, headers);
            }
        }
    }

    /**
     * 缓存服务端上报的监测
     *
     * @param urls
     */
    public void saveExceed(ReportType type, String bidId, List<String> urls) {
        if (CollectionUtils.isEmpty(urls)) {
            return;
        }
        try (Jedis jedis = jedisPool.getResource()) {
            try (Pipeline pipeline = jedis.pipelined()) {
                String key = getKeyGenerator(type, bidId).generateKey();
                pipeline.select(serverReportProperties.getQueueDb());
                pipeline.lpush(key, urls.toArray(new String[0]));
                pipeline.expire(key, serverReportProperties.getExpire());
                List<Object> response = pipeline.syncAndReturnAll();
                if (log.isDebugEnabled()) {
                    log.debug("pipeline response {}", response);
                }
            }
        } catch (Exception e) {
            log.error("save exceed err {}, affect {}", e.getMessage(), urls);
        }
    }

    /**
     * 获取存储的key
     *
     * @param type
     * @param id
     * @return
     */
    public KeyGenerator getKeyGenerator(ReportType type, String id) {
        return CacheKeys.ServerReport.getServerReportTempKey(type.name(), id);
    }

    public ServerReportConfig getConfig(String mediaId, String appId, String dspId, String bidId) {
        ServerReportProperties.MediaConfig mediaConfig = getMedia(mediaId, appId);
        if (Objects.isNull(mediaConfig)) {
            return null;
        }
        if (!serverReportProperties.getDspIds().contains(dspId)) {
            return null;
        }

        return ServerReportConfig.builder()
                .limit(mediaConfig.getNum() - 1)
                .bidId(bidId)
                .domains(serverReportProperties.getDomains())
                .build();
    }

    /**
     * 匹配媒体
     *
     * @param mediaId
     * @param appId
     * @return
     */
    private ServerReportProperties.MediaConfig getMedia(String mediaId, String appId) {
        List<ServerReportProperties.MediaConfig> media = serverReportProperties.getMedia();
        for (ServerReportProperties.MediaConfig config : media) {
            if (mediaId.equals(config.getId())) {
                List<String> appIds = config.getAppIds();
                if (CollectionUtils.isNotEmpty(appIds)) {
                    if (appIds.contains(appId)) {
                        return config;
                    }
                } else {
                    return config;
                }
            }
        }
        return null;
    }

    public BasicHeader[] convertToBasicHeaders(MultiValueMap<String, String> map) {
        BasicHeader[] headers = new BasicHeader[map.size()];
        int i = 0;
        for (String key : map.keySet()) {
            String[] values = map.get(key).toArray(new String[0]);
            headers[i] = new BasicHeader(key, String.join(", ", values));
            i++;
        }
        return headers;
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    static class Request {
        private String url;
        private BasicHeader[] headers;
    }
}
