package com.bxm.adx.common.risk;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.User;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.Builder;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.openxml4j.util.ZipSecureFile;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Async;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.util.*;

/**
 * 是风控的临时需求 只会存在几天 所以把所有的逻辑都放在里面 后期要删除可以把配置删掉
 *
 * @author zhangdong
 * @date 2022/11/28
 */
@RestController
@RequestMapping("/risk")
@ConditionalOnProperty(prefix = "adx", name = "risk.crowd-hit-open", havingValue = "true")
public class RiskCrowdHit {

    @Autowired
    private Counter counter;
    @Autowired
    private Fetcher fetcher;

    private Map<String, Set<String>> map = new HashMap<>();
    private final String ip = "IP";
    private final String device = "DEVICEID";
    private final String uid = "UID";
    private final String token = "riskCrowdHitQuery";
    private final Integer expireTime = 3600 * 24 * 14;


    @PostConstruct
    public void init() {
        map.put(ip, new HashSet<>());
        map.put(device, new HashSet<>());
        map.put(uid, new HashSet<>());
        ZipSecureFile.setMinInflateRatio(-1);
        ExcelReader reader = ExcelUtil.getReader(FileUtil.file("classpath:risk/ADX风控包.xlsx"));
        int sheetCount = reader.getSheetCount();
        reader.close();
        for (int i = 0; i < sheetCount; i++) {
            readSheet(ExcelUtil.getReader(FileUtil.file("classpath:risk/ADX风控包.xlsx"), i));
        }
    }

    private void readSheet(ExcelReader reader) {
        try {
            List<RiskCrowdData> crowdData = reader.readAll(RiskCrowdData.class);
            for (RiskCrowdData data : crowdData) {
                Set<String> contents = map.get(data.getType());
                if (contents != null && StringUtils.isNotEmpty(data.getContent())) {
                    contents.add(data.getContent());
                }
            }
        } finally {
            reader.close();
        }
    }

    @GetMapping(value = "/crowd/hit")
    public List<RiskCrowdHitData> getCrowdHitData(@RequestParam("token") String token) {
        if (!this.token.equals(token)) {
            return null;
        }
        //获取前后五天的数据
        List<RiskCrowdHitData> dataList = Lists.newArrayList();
        DateTime dateTime = DateUtil.offsetDay(DateUtil.date(), -5);
        for (int i = 1; i <= 10; i++) {
            String tmp = DateUtil.offsetDay(dateTime, i).toString("yyyy-MM-dd");
            Long requestCount = fetcher.fetch(AdxKeyGenerator.Risk.getRiskCrowdRequestCount(tmp), Long.class);
            if (requestCount == null) {
                continue;
            }
            Long hitCount = fetcher.fetch(AdxKeyGenerator.Risk.getRiskCrowdHitCount(tmp), Long.class);
            Double hitRate = null;
            if (hitCount != null) {
                hitRate = (double) hitCount / (double) requestCount;
            }
            dataList.add(RiskCrowdHitData.builder().date(tmp).requestCount(requestCount).crowdHitCount(hitCount).hitRate(hitRate).build());
        }
        return dataList;
    }

    @Async("taskExecutor")
    public void crowdHitDeal(BidRequest request) {
        if (request == null) {
            return;
        }
        //增加请求数值
        String date = DateUtil.today();
        redisIncrease(date, true);
        //根据三个维度判断是否命中
        boolean hitIncrease = false;
        User user = request.getUser();
        if (user != null) {
            hitIncrease = map.get(uid).contains(user.getId());
        }
        Device device = request.getDevice();
        if (!hitIncrease && device != null) {
            hitIncrease = map.get(ip).contains(device.getIp());
            if (!hitIncrease) {
                List<String> deviceIdList = Lists.newArrayList(device.getImei(), device.getImei_md5(), device.getIdfa(), device.getIdfa_md5(),
                        device.getMac(), device.getMac_md5(), device.getOaid(), device.getOaid_md5(), device.getDpid(), device.getDpid_md5(), device.getVaid(), device.getGaid(), device.getGaid_md5());
                for (String tmp : deviceIdList) {
                    if (hitIncrease = map.get(this.device).contains(tmp)) {
                        break;
                    }
                }
            }
        }
        //命中 增加命中数值
        if (hitIncrease) {
            redisIncrease(date, false);
        }
    }


    private void redisIncrease(String date, boolean isRequest) {
        if (isRequest) {
            KeyGenerator requestCount = AdxKeyGenerator.Risk.getRiskCrowdRequestCount(date);
            counter.incrementAndGet(requestCount, expireTime);
        } else {
            KeyGenerator hitCount = AdxKeyGenerator.Risk.getRiskCrowdHitCount(date);
            counter.incrementAndGet(hitCount, expireTime);
        }
    }


    @Data
    static class RiskCrowdData {

        private String content;

        private String type;
    }

    @Data
    @Builder
    static class RiskCrowdHitData {
        private String date;
        private Long requestCount;
        private Long crowdHitCount;
        private Double hitRate;
    }
}
