package com.bxm.adx.service.common.sifter.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.rule.task.TaskAppRuleSo;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 开发者应用定向<br/>
 *
 * @author kerry.jiang
 * @date 2020/6/1 14:37
 */
@Component
@Slf4j
public class TaskAppSifter extends TaskAbstractSifter {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    
    @Override
    public boolean filter(Context<TaskData> context) {
        if(StringUtils.isEmpty(context.getData().getRequest().getAppId())){
            return false;
        }
        //填充应用黑白名单信息
        fillApp(context);
        
        //应用黑白名单限制
        if(filterApp(context)){
            return true;
        }
        
        return false;
    }

    /**
     * 应用黑白名单限制
     */
    private boolean filterApp(Context<TaskData> context) {
        String appId = context.getData().getRequest().getAppId();
        TaskCache data = context.getData().getTaskCache();
        TaskAppRuleSo rule = context.getData().getMapAppRule().get(data.getId());
        if(null == rule || CollectionUtils.isEmpty(rule.getApps())){
            return false;
        }

        boolean blackList = rule.isBlackList();
        Set<String> list = rule.getApps();
        if(blackList){//黑名单
            if(list.contains(appId)){
                super.addReason(context, "在开发者应用黑名单");
                return true;
            }
        }else{//白名单
            if(!list.contains(appId)){
                super.addReason(context, "不在开发者应用白名单");
                return true;
            }
        }
        return false;
    }

    /**
     * 填充应用黑白名单信息
     */
    private void fillApp(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagApp())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }

        Map<Long, TaskAppRuleSo> mapAppRule = new HashMap<>();
        data.setFlagApp(Boolean.TRUE);//取值标识变更
        data.setMapAppRule(mapAppRule);

        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<String>> mapAppRuleRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapAppRuleRs.put(id, pipeline.get(AdxKeyGenerator.Task.getAppRule(id).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillApp error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充广告位黑白名单信息
        fillApp(mapAppRule, mapAppRuleRs);
    }

    /**
     * 填充广告位黑白名单信息
     */
    private void fillApp(Map<Long, TaskAppRuleSo> mapAppRule, Map<Long, Response<String>> mapAppRuleRs) {
        Set<Long> keySet = mapAppRuleRs.keySet();
        for (Long id : keySet) {
            Response<String> strRs = mapAppRuleRs.get(id);
            String str = strRs.get();//黑白名单信息
            if(StringUtils.isBlank(str)){
                continue;
            }
            mapAppRule.put(id, JSON.parseObject(str, TaskAppRuleSo.class));
        }
    }
    
}
