package com.bxm.adx.service.common.sifter.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.model.task.TaskInfoRequest;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 地域定向<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/26 20:40
 */
@Component
@Slf4j
public class TaskRegionSifter extends TaskAbstractSifter {

    /**
     * 如果这个ip无法得到行政编码，默认一个无效的编码
     */
    private static final String DEFAULT_CODE = "000000";

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    
    @Override
    public boolean filter(Context<TaskData> context) {
        //填充地域信息
        fillRegion(context);

        //地域限制
        if(filterRegion(context)){
            return true;
        }
        
        return false;
    }

    /**
     * 地域限制
     */
    private boolean filterRegion(Context<TaskData> context) {
        TaskInfoRequest request = context.getData().getRequest();
        TaskCache data = context.getData().getTaskCache();
        Set<String> regionCodes = context.getData().getMapRegion().get(data.getId());
        if(CollectionUtils.isEmpty(regionCodes)){
            return false;
        }

        String regionCode = StringUtils.defaultIfEmpty(request.getIprst().getRegioncode(), DEFAULT_CODE);
        boolean flagSelf = regionCodes.contains(regionCode);
        boolean flagProvince = regionCodes.contains(getProvinceCode(regionCode));
        if(!flagSelf && !flagProvince){
            super.addReason(context, "地域限制");
            return true;
        }
        return false;
    }

    /**
     * 所属省级代码
     */
    private String getProvinceCode(String regionCode) {
        return StringUtils.rightPad(StringUtils.left(regionCode, 2), 6, "0");
    }

    /**
     * 填充地域信息
     */
    private void fillRegion(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagRegion())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }

        Map<Long, Set<String>> mapRegion = new HashMap<>();
        data.setFlagRegion(Boolean.TRUE);//取值标识变更
        data.setMapRegion(mapRegion);
        
        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<String>> mapRegionRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapRegionRs.put(id, pipeline.get(AdxKeyGenerator.Task.getRegionList(id).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillRegion error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充地域信息
        fillRegion(mapRegion, mapRegionRs);
    }

    /**
     * 填充地域信息
     */
    private void fillRegion(Map<Long, Set<String>> mapRegion, Map<Long, Response<String>> mapRegionRs) {
        Set<Long> keySet = mapRegionRs.keySet();
        for (Long id : keySet) {
            Response<String> strRs = mapRegionRs.get(id);
            String str = strRs.get();//地域列表
            if(StringUtils.isBlank(str)){
                continue;
            }
            mapRegion.put(id, JSON.parseObject(str, Set.class));
        }
    }
}
