package com.bxm.adx.service.common.sifter.task;

import java.util.*;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.adx.facade.rule.task.TaskTimeSlotRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 投放时段限制<br/>
 *
 * @author kerry.jiang
 * @date 2020/7/13 11:50
 */
@Component
@Slf4j
public class TaskTimeSlotSifter extends TaskAbstractSifter {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Override
    public boolean filter(Context<TaskData> context) {
        //填充投放时段信息
        fillTimeSlot(context);

        //投放时段限制
        if(filterTimeSlot(context)){
            return true;
        }

        return false;
    }

    /**
     * 投放时段限制
     */
    private boolean filterTimeSlot(Context<TaskData> context) {
        TaskCache data = context.getData().getTaskCache();
        TaskTimeSlotRuleSo rule = context.getData().getMapTimeSlotRule().get(data.getId());
        if(null == rule || MapUtils.isEmpty(rule.getWeekTimeSlot())){
            return false;
        }
        TaskChainData chainData = context.getData().getChainData();

        int weekDayIndex = getWeekDay(chainData.getToday()) - 1;
        Map<Integer, Boolean> timeSlot = rule.getWeekTimeSlot().get(weekDayIndex);
        if(MapUtils.isEmpty(timeSlot)){
            return false;
        }
        //小时0~23
        int hours = chainData.getToday().getHours();
        Boolean flag = timeSlot.get(hours);
        if(null == flag || !flag){
            //不在投放时段
            super.addReason(context, "不在投放时段");
            return true;
        }
        return false;
    }

    /**
     * 周几，周一：1 ... 周日：7
     */
    private int getWeekDay(Date date){
        Calendar cld = Calendar.getInstance(Locale.CHINA);
        cld.setTime(date);
        int days = cld.get(Calendar.DAY_OF_WEEK);
        if(days == 1){
            return 7;
        }else{
            return days - 1;
        }
    }

    /**
     * 填充投放时段信息
     */
    private void fillTimeSlot(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagTimeSlot())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }

        Map<Long, TaskTimeSlotRuleSo> mapTimeSlotRule = new HashMap<>();
        data.setFlagTimeSlot(Boolean.TRUE);//取值标识变更
        data.setMapTimeSlotRule(mapTimeSlotRule);

        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<String>> mapTimeSlotRuleRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapTimeSlotRuleRs.put(id, pipeline.get(AdxKeyGenerator.Task.getTimeSlotRule(id).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillTimeSlot error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充投放时段信息
        fillTimeSlot(mapTimeSlotRule, mapTimeSlotRuleRs);
    }

    /**
     * 填充投放时段信息
     */
    private void fillTimeSlot(Map<Long, TaskTimeSlotRuleSo> mapTimeSlotRule, Map<Long, Response<String>> mapTimeSlotRuleRs) {
        Set<Long> keySet = mapTimeSlotRuleRs.keySet();
        for (Long id : keySet) {
            Response<String> strRs = mapTimeSlotRuleRs.get(id);
            String str = strRs.get();//时段信息
            if(StringUtils.isBlank(str)){
                continue;
            }
            mapTimeSlotRule.put(id, JSON.parseObject(str, TaskTimeSlotRuleSo.class));
        }
    }
}
