package com.bxm.adx.service.common.pushable;

import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.rule.task.*;
import com.bxm.adx.service.common.convert.TaskConvert;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Converter;
import com.bxm.warcar.cache.push.HashUpdating;
import com.bxm.warcar.cache.push.JSONObjectPushable;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * 推送接收器-任务信息<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/26 11:19
 */
@Component
@CachePush("TASK")
@Slf4j
public class TaskPushable extends JSONObjectPushable<TaskOfRules> implements HashUpdating<TaskOfRules> {

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Autowired @Qualifier("taskRegionPushable")
    private Pushable taskRegionPushable;
    @Autowired @Qualifier("taskAppPushable")
    private Pushable taskAppPushable;
    @Autowired @Qualifier("taskPositionPushable")
    private Pushable taskPositionPushable;
    @Autowired @Qualifier("taskFrequencyPushable")
    private Pushable taskFrequencyPushable;
    @Autowired @Qualifier("taskOpenedPushable")
    private Pushable taskOpenedPushable;
    @Autowired @Qualifier("taskDeviceBrandPushable")
    private Pushable taskDeviceBrandPushable;
    @Autowired @Qualifier("taskTimeSlotPushable")
    private Pushable taskTimeSlotPushable;

    public TaskPushable() {
        super(new Converter<TaskOfRules>() {
            @Override
            public Object convert(TaskOfRules object) {
                return TaskConvert.of(object);
            }
        });
    }

    @Override
    protected Class<TaskOfRules> getClsType() {
        return TaskOfRules.class;
    }

    @Override
    public Updater getUpdater() {
        return updater;
    }

    @Override
    public String getField(Map<String, Object> parameters, TaskOfRules object) {
        return String.valueOf(object.getId());
    }

    @Override
    protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
        return AdxKeyGenerator.Task.getAllList();
    }

    @Override
    protected void beforeUpdate(Map<String, Object> parameters, byte[] data, TaskOfRules object) {
//        if(log.isDebugEnabled()){
//            log.debug("TaskOfRules={}", object);
//        }
        //地域
        Rule region = object.getRegion();
        taskRegionPushable.push(parameters, null == region ? null : JsonHelper.convert2bytes(region));
        
        //开发者应用
        Rule app = object.getApps();
        taskAppPushable.push(parameters, null == app ? null : JsonHelper.convert2bytes(app));
        
        //广告位
        Rule position = object.getPosition();
        taskPositionPushable.push(parameters, null == position ? null : JsonHelper.convert2bytes(position));
        
        //频次
        Rule frequencys = object.getFrequencys();
        taskFrequencyPushable.push(parameters, null == frequencys ? null : JsonHelper.convert2bytes(frequencys));
        
        //设备品牌名
        Rule deviceBrands = object.getDeviceBrands();
        taskDeviceBrandPushable.push(parameters, null == deviceBrands ? null : JsonHelper.convert2bytes(deviceBrands));

        //投放时段
        Rule timeSlots = object.getTimeSlots();
        taskTimeSlotPushable.push(parameters, null == timeSlots ? null : JsonHelper.convert2bytes(timeSlots));

        //开放中的任务
        taskOpenedPushable.push(parameters, data);
    }
    
    /**
     * 开发者应用<br/>
     */
    @Bean
    public Pushable taskAppPushable() {
        return new JSONObjectPushable<TaskAppRuleSo>() {

            @Override
            protected Class<TaskAppRuleSo> getClsType() {
                return TaskAppRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                Object taskId = parameters.get(CachePushableFields.TASK_ID);
                return AdxKeyGenerator.Task.getAppRule(TypeHelper.castToLong(taskId));
            }
        };
    }
    
    /**
     * 广告位<br/>
     */
    @Bean
    public Pushable taskPositionPushable() {
        return new JSONObjectPushable<TaskPositionRuleSo>() {

            @Override
            protected Class<TaskPositionRuleSo> getClsType() {
                return TaskPositionRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                Object taskId = parameters.get(CachePushableFields.TASK_ID);
                return AdxKeyGenerator.Task.getPositionRule(TypeHelper.castToLong(taskId));
            }
        };
    }
    
    /**
     * 频次<br/>
     */
    @Bean
    public Pushable taskFrequencyPushable() {
        return new JSONObjectPushable<TaskFrequencyRuleSo>() {

            @Override
            protected Class<TaskFrequencyRuleSo> getClsType() {
                return TaskFrequencyRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                Object taskId = parameters.get(CachePushableFields.TASK_ID);
                return AdxKeyGenerator.Task.getFrequencyRule(TypeHelper.castToLong(taskId));
            }
        };
    }
    
    /**
     * 设备品牌名<br/>
     */
    @Bean
    public Pushable taskDeviceBrandPushable() {
        return new JSONObjectPushable<TaskDeviceBrandRuleSo>() {

            @Override
            protected Class<TaskDeviceBrandRuleSo> getClsType() {
                return TaskDeviceBrandRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                Object taskId = parameters.get(CachePushableFields.TASK_ID);
                return AdxKeyGenerator.Task.getDeviceBrandRule(TypeHelper.castToLong(taskId));
            }
        };
    }

    /**
     * 投放时段限制<br/>
     */
    @Bean
    public Pushable taskTimeSlotPushable() {
        return new JSONObjectPushable<TaskTimeSlotRuleSo>() {

            @Override
            protected Class<TaskTimeSlotRuleSo> getClsType() {
                return TaskTimeSlotRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                Object taskId = parameters.get(CachePushableFields.TASK_ID);
                return AdxKeyGenerator.Task.getTimeSlotRule(TypeHelper.castToLong(taskId));
            }
        };
    }
}
