package com.bxm.adx.service.common.sifter.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.constant.redis.TaskCounterField;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 统计数限制<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/26 20:40
 */
@Component
@Slf4j
public class TaskCounterSifter extends TaskAbstractSifter {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    
    @Override
    public boolean filter(Context<TaskData> context) {
        //填充任务统计信息
        fillCounter(context);

        //总统计数限制
        if(filterTotal(context)){
            return true;
        }
        
        //今日统计数限制
        if(filterToday(context)){
            return true;
        }

        return false;
    }

    /**
     * 总统计数限制
     */
    private boolean filterTotal(Context<TaskData> context) {
        TaskCache data = context.getData().getTaskCache();
        Map<String, String> map = context.getData().getMapCtTotal().get(data.getId());
        if(MapUtils.isEmpty(map)){
            return false;
        }
        Long views = NumberUtils.toLong(map.get(TaskCounterField.VIEWS), 0);
        Long clicks = NumberUtils.toLong(map.get(TaskCounterField.CLICKS), 0);
        Long wakes = NumberUtils.toLong(map.get(TaskCounterField.WAKES), 0);
        if(null == data.getViewTotalLimit() || data.getViewTotalLimit() < 1){
            //不限
        }else if(views >= data.getViewTotalLimit()){
            super.addReason(context, "总曝光限制");
            return true;
        }
        if(null == data.getClickTotalLimit() || data.getClickTotalLimit() < 1){
            //不限
        }else if(clicks >= data.getClickTotalLimit()){
            super.addReason(context, "总点击限制");
            return true;
        }
        if(null == data.getWakeUpTotalLimit() || data.getWakeUpTotalLimit() < 1){
            //不限
        }else if(wakes >= data.getWakeUpTotalLimit()){
            super.addReason(context, "总唤醒限制");
            return true;
        }
        return false;
    }

    /**
     * 今日统计数限制
     */
    private boolean filterToday(Context<TaskData> context) {
        TaskCache data = context.getData().getTaskCache();
        Map<String, String> map = context.getData().getMapCtToday().get(data.getId());
        if(MapUtils.isEmpty(map)){
            return false;
        }
        Long views = NumberUtils.toLong(map.get(TaskCounterField.VIEWS), 0);
        Long clicks = NumberUtils.toLong(map.get(TaskCounterField.CLICKS), 0);
        Long wakes = NumberUtils.toLong(map.get(TaskCounterField.WAKES), 0);
        if(null == data.getViewDayLimit() || data.getViewDayLimit() < 1){
            //不限
        }else if(views >= data.getViewDayLimit()){
            super.addReason(context, "日曝光限制");
            return true;
        }
        if(null == data.getClickDayLimit() || data.getClickDayLimit() < 1){
            //不限
        }else if(clicks >= data.getClickDayLimit()){
            super.addReason(context, "日点击限制");
            return true;
        }
        if(null == data.getWakeUpDayLimit() || data.getWakeUpDayLimit() < 1){
            //不限
        }else if(wakes >= data.getWakeUpDayLimit()){
            super.addReason(context, "日唤醒限制");
            return true;
        }
        return false;
    }

    /**
     * 填充任务统计信息
     */
    private void fillCounter(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagCounter())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        TaskChainData chainData = context.getData().getChainData();
        
        Map<Long, Map<String, String>> mapCtTotal = new HashMap<>();
        Map<Long, Map<String, String>> mapCtToday = new HashMap<>();
        data.setFlagCounter(Boolean.TRUE);//取值标识变更
        data.setMapCtTotal(mapCtTotal);
        data.setMapCtToday(mapCtToday);
        
        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<Map<String, String>>> mapCtTotalRs = new HashMap<>();
        Map<Long, Response<Map<String, String>>> mapCtTodayRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapCtTotalRs.put(id, pipeline.hgetAll(AdxKeyGenerator.Task.getCounter(id).generateKey()));
                mapCtTodayRs.put(id, pipeline.hgetAll(AdxKeyGenerator.Task.getCounter(id, chainData.getTodayYmd()).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillCounter error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充任务统计信息
        fillCounter(mapCtTotal, mapCtToday, mapCtTotalRs, mapCtTodayRs);
    }

    /**
     * 填充任务统计信息
     */
    private void fillCounter(Map<Long, Map<String, String>> mapCtTotal, Map<Long, Map<String, String>> mapCtToday,
                             Map<Long, Response<Map<String, String>>> mapCtTotalRs,
                             Map<Long, Response<Map<String, String>>> mapCtTodayRs) {
        Set<Long> keySet = mapCtTotalRs.keySet();
        for (Long id : keySet) {
            Response<Map<String, String>> mapRs = mapCtTotalRs.get(id);
            Map<String, String> map = mapRs.get();//统计信息
            if(MapUtils.isEmpty(map)){
                continue;
            }
            mapCtTotal.put(id, map);
        }
        keySet = mapCtTodayRs.keySet();
        for (Long id : keySet) {
            Response<Map<String, String>> mapRs = mapCtTodayRs.get(id);
            Map<String, String> map = mapRs.get();//统计信息
            if(MapUtils.isEmpty(map)){
                continue;
            }
            mapCtToday.put(id, map);
        }
    }
}
