package com.bxm.adx.service.common.sifter.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.constant.redis.TaskCounterField;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.adx.facade.rule.task.TaskFrequencyRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 频次限制<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/26 20:40
 */
@Component
@Slf4j
public class TaskFrequencySifter extends TaskAbstractSifter {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    
    @Override
    public boolean filter(Context<TaskData> context) {
        //填充任务频次信息
        fillFrequency(context);

        //设备频次限制
        if(filterFrequency(context)){
            return true;
        }

        return false;
    }

    /**
     * 设备频次限制
     */
    private boolean filterFrequency(Context<TaskData> context) {
        TaskCache data = context.getData().getTaskCache();
        Map<String, String> map = context.getData().getMapCtTodayDev().get(data.getId());
        TaskFrequencyRuleSo rule = context.getData().getMapFreqRule().get(data.getId());
        if(MapUtils.isEmpty(map) || null == rule){
            return false;
        }
        Long views = NumberUtils.toLong(map.get(TaskCounterField.VIEWS), 0);
        Long clicks = NumberUtils.toLong(map.get(TaskCounterField.CLICKS), 0);
        Long wakes = NumberUtils.toLong(map.get(TaskCounterField.WAKES), 0);

        List<TaskFrequencyRuleSo.Entry> entries = rule.getEntries();
        for (TaskFrequencyRuleSo.Entry entry : entries) {
            if(TaskFrequencyRuleSo.Entry.CYCLE_EVERYDAY != entry.getCycle()){//仅支持每天
                continue;
            }
            if(TaskFrequencyRuleSo.Entry.TYPE_VIEW == entry.getType()){
                if(views >= entry.getCount()){
                    super.addReason(context, "单设备日曝光限制");
                    return true;
                }
            }else if(TaskFrequencyRuleSo.Entry.TYPE_CLICK == entry.getType()){
                if(clicks >= entry.getCount()){
                    super.addReason(context, "单设备日点击限制");
                    return true;
                }
            }else if(TaskFrequencyRuleSo.Entry.TYPE_WAKE == entry.getType()){
                if(wakes >= entry.getCount()){
                    super.addReason(context, "单设备日唤醒限制");
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 填充任务频次信息
     */
    private void fillFrequency(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagFrequency())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        TaskChainData chainData = context.getData().getChainData();
        
        Map<Long, Map<String, String>> mapCtTodayDev = new HashMap<>();
        Map<Long, TaskFrequencyRuleSo> mapFreqRule = new HashMap<>();
        data.setFlagFrequency(Boolean.TRUE);//取值标识变更
        data.setMapCtTodayDev(mapCtTodayDev);
        data.setMapFreqRule(mapFreqRule);
        
        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<Map<String, String>>> mapCtTodayDevRs = new HashMap<>();
        Map<Long, Response<String>> mapFreqRuleRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapCtTodayDevRs.put(id, pipeline.hgetAll(AdxKeyGenerator.Task.getCounter(id, chainData.getTodayYmd(),
                        data.getRequest().getDevUid()).generateKey()));
                mapFreqRuleRs.put(id, pipeline.get(AdxKeyGenerator.Task.getFrequencyRule(id).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillFrequency error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充任务频次信息
        fillFrequency(mapCtTodayDev, mapFreqRule, mapCtTodayDevRs, mapFreqRuleRs);
    }

    /**
     * 填充任务频次信息
     */
    private void fillFrequency(Map<Long, Map<String, String>> mapCtTodayDev, Map<Long, TaskFrequencyRuleSo> mapFreqRule,
                             Map<Long, Response<Map<String, String>>> mapCtTodayDevRs,
                             Map<Long, Response<String>> mapFreqRuleRs) {
        Set<Long> keySet = mapCtTodayDevRs.keySet();
        for (Long id : keySet) {
            Response<Map<String, String>> mapRs = mapCtTodayDevRs.get(id);
            Map<String, String> map = mapRs.get();//单设备每日统计信息
            if(MapUtils.isEmpty(map)){
                continue;
            }
            mapCtTodayDev.put(id, map);
        }
        keySet = mapFreqRuleRs.keySet();
        for (Long id : keySet) {
            Response<String> strRs = mapFreqRuleRs.get(id);
            String str = strRs.get();//频次限制信息
            if(StringUtils.isBlank(str)){
                continue;
            }
            mapFreqRule.put(id, JSON.parseObject(str, TaskFrequencyRuleSo.class));
        }
    }
}
