package com.bxm.adx.service.common.sifter.task;

import java.util.*;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.model.position.Position;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.adx.facade.rule.task.TaskPositionRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.sifter.Context;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 广告位定向<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/26 20:40
 */
@Component
@Slf4j
public class TaskPositionSifter extends TaskAbstractSifter {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    
    @Override
    public boolean filter(Context<TaskData> context) {
        //填充广告位黑白名单信息
        fillPosition(context);
        
        //广告位黑白名单限制
        if(filterPosition(context)){
            return true;
        }
        
        return false;
    }

    /**
     * 广告位黑白名单限制
     */
    private boolean filterPosition(Context<TaskData> context) {
        String positionId = context.getData().getRequest().getPositionId();
        String appId = context.getData().getRequest().getAppId();
        TaskCache data = context.getData().getTaskCache();
        TaskPositionRuleSo rule = context.getData().getMapPositionRule().get(data.getId());
        if(null == rule || CollectionUtils.isEmpty(rule.getPositions())){
            return false;
        }

        boolean blackList = rule.isBlackList();
        Set<String> list = rule.getPositions();
        if(StringUtils.isNotEmpty(positionId)){//老接口
            if(blackList){//黑名单
                if(list.contains(positionId)){
                    super.addReason(context, "在广告位黑名单");
                    return true;
                }
            }else{//白名单
                if(!list.contains(positionId)){
                    super.addReason(context, "不在广告位白名单");
                    return true;
                }
            }
        }else if(StringUtils.isNotEmpty(appId)){//新接口
            //兼容开发者应用
            Set<String> appIds = getAppIds(list);
            if(blackList){//黑名单
                if(appIds.contains(appId)){
                    super.addReason(context, "在开发者应用黑名单");
                    return true;
                }
            }else{//白名单
                if(!appIds.contains(appId)){
                    super.addReason(context, "不在开发者应用白名单");
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 获取开发者应用appid列表
     */
    private Set<String> getAppIds(Set<String> positionIds){
        Set<String> appIds = new HashSet<>();
        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        List<Response<String>> listAppRs = new ArrayList<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for(String positionId : positionIds){
                listAppRs.add(pipeline.get(AdxKeyGenerator.Position.getInfo(positionId).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("getAppIds error: positionIds={},e={}",
                    positionIds, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
        for (Response<String> strRs : listAppRs) {
            String str = strRs.get();//应用信息
            if(StringUtils.isBlank(str)){
                continue;
            }
            Position position = JSON.parseObject(str, Position.class);
            if(null != position){
                appIds.add(position.getAppId());
            }
        }
        return appIds;
    }

    /**
     * 填充广告位黑白名单信息
     */
    private void fillPosition(Context<TaskData> context) {
        TaskData data = context.getData();
        if(Boolean.TRUE.equals(data.getFlagPosition())){//取值标识判断
            return ;
        }
        List<Long> ids = super.getIds(context);
        if(CollectionUtils.isEmpty(ids)){
            return;
        }

        Map<Long, TaskPositionRuleSo> mapPositionRule = new HashMap<>();
        data.setFlagPosition(Boolean.TRUE);//取值标识变更
        data.setMapPositionRule(mapPositionRule);

        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<String>> mapPositionRuleRs = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (Long id : ids) {
                mapPositionRuleRs.put(id, pipeline.get(AdxKeyGenerator.Task.getPositionRule(id).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
            log.error("fillPosition error: ids={},e={}",
                    ids, ExceptionUtils.getFullStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //填充广告位黑白名单信息
        fillPosition(mapPositionRule, mapPositionRuleRs);
    }

    /**
     * 填充广告位黑白名单信息
     */
    private void fillPosition(Map<Long, TaskPositionRuleSo> mapPositionRule, Map<Long, Response<String>> mapPositionRuleRs) {
        Set<Long> keySet = mapPositionRuleRs.keySet();
        for (Long id : keySet) {
            Response<String> strRs = mapPositionRuleRs.get(id);
            String str = strRs.get();//黑白名单信息
            if(StringUtils.isBlank(str)){
                continue;
            }
            mapPositionRule.put(id, JSON.parseObject(str, TaskPositionRuleSo.class));
        }
    }
    
}
