package com.bxm.adx.service.facade;

import java.util.*;
import java.util.regex.Pattern;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.adx.facade.constant.Constants;
import com.bxm.adx.facade.constant.enums.AppPlatformTypeEnum;
import com.bxm.adx.facade.constant.enums.DevOsEnum;
import com.bxm.adx.facade.constant.enums.TaskOpsceneEnum;
import com.bxm.adx.facade.constant.enums.TaskTypeEnum;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.constant.redis.TaskCounterField;
import com.bxm.adx.facade.model.dev.DevRequest;
import com.bxm.adx.facade.model.position.Position;
import com.bxm.adx.facade.model.task.TaskCounter;
import com.bxm.adx.facade.model.task.TaskInfo;
import com.bxm.adx.facade.model.task.TaskInfoRequest;
import com.bxm.adx.facade.rule.app.App;
import com.bxm.adx.facade.rule.task.TaskCache;
import com.bxm.adx.facade.service.TaskService;
import com.bxm.adx.service.common.sifter.task.TaskSifterForListChain;
import com.bxm.adx.service.common.sifter.task.TaskSifterForSingleChain;
import com.bxm.adx.service.common.util.DevHelper;
import com.bxm.adx.service.integration.TaskCounterIntegration;
import com.bxm.adx.service.service.IpService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.id.IdGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.StringHelper;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * 任务信息<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/25 20:09
 */
@RestController
@Slf4j
public class TaskServiceImpl implements TaskService {
    
    private final static Pattern pattern = Pattern.compile(".*/task/+countInfo.*");

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    private IdGenerator idGenerator;

    @Autowired
    private IpService ipService;
    @Autowired
    private TaskSifterForSingleChain taskSifterForSingleChain;
    @Autowired
    private TaskSifterForListChain taskSifterForListChain;
    @Autowired
    private TaskCounterIntegration taskCounterIntegration;

    @Override
    public TaskInfo getTask(@RequestBody TaskInfoRequest request) {
        if(log.isDebugEnabled()){
            log.debug("request={}", JsonHelper.convert(request));
        }
        Position position = fetcher.fetch(AdxKeyGenerator.Position.getInfo(request.getPositionId()), Position.class);
        if(null == position){
            log.warn("position is null, positionId={}", request.getPositionId());
            return null;
        }
        request.setAppId(position.getAppId());
        //应用是否有开启耍呗任务
        if (!checkApp(request)){
            return null;
        }
        //任务请求埋点
        String bidId = idGenerator.next();
        taskCounterIntegration.counterRequest(request, bidId);
        //开放中的任务
        List<TaskCache> taskCacheList = getOpenedList();
        if (CollectionUtils.isEmpty(taskCacheList)) {
            if (log.isDebugEnabled()) {
                log.debug("taskCacheList is empty!");
            }
            return null;
        }

        //自动填充参数
        autoFillTaskInfoRequest(request);

        //筛选过滤
        TaskCache taskCache = taskSifterForSingleChain.filter(request, taskCacheList);
        if(null == taskCache){
            return null;
        }
        //任务响应埋点
        taskCounterIntegration.counterResponse(request, bidId, taskCache.getId());
        return getTaskInfo(request, taskCache, bidId);
    }

    @Override
    public List<TaskInfo> getTasks(@RequestBody TaskInfoRequest request) {
        if(log.isDebugEnabled()){
            log.debug("request={}", JsonHelper.convert(request));
        }
        //应用是否有开启耍呗任务
        if (!checkApp(request)){
            return null;
        }
        //开放中的任务
        List<TaskCache> taskCacheList = getOpenedList();
        if (CollectionUtils.isEmpty(taskCacheList)) {
            if (log.isDebugEnabled()) {
                log.debug("taskCacheList is empty!");
            }
            //任务请求埋点
            taskCounterIntegration.counterRequest(request, idGenerator.next());
            return null;
        }

        //自动填充参数
        autoFillTaskInfoRequest(request);
        
        //筛选过滤
        List<TaskCache> taskCaches = taskSifterForListChain.filter(request, taskCacheList);
        if(CollectionUtils.isEmpty(taskCaches)){
            //任务请求埋点
            taskCounterIntegration.counterRequest(request, idGenerator.next());
            return null;
        }
        return getTaskInfos(request, taskCaches);
    }

    /**
     * 获取任务列表
     */
    private List<TaskInfo> getTaskInfos(TaskInfoRequest request, List<TaskCache> taskCaches) {
        List<TaskInfo> resps = new ArrayList<>();
        String bidId = null;
        for(TaskCache taskCache : taskCaches){
            if(null != request.getOpscene()){
                if(TaskOpsceneEnum.SILENCE_WAKE.getId() == request.getOpscene()){
                    //只出静默唤醒任务
                    if(taskCache.getType() != TaskTypeEnum.SILENCE_WAKE.getId()){
                        continue;
                    }
                }else if(TaskOpsceneEnum.EXIT.getId() == request.getOpscene()){
                    //只出复制任务
                    if(taskCache.getType() != TaskTypeEnum.COPY.getId()){
                        continue;
                    }
                }
            }

            if(taskCache.getType() == TaskTypeEnum.COPY.getId()){//复制任务
                //最多缓存1天
                updater.update(AdxKeyGenerator.Counter.getTaskAcquireTime(taskCache.getId(), request.getDevUid()),
                        System.currentTimeMillis(), 86400);
            }

            bidId = idGenerator.next();
            doTracker(request, resps, taskCache, bidId);
            resps.add(getTaskInfo(request, taskCache, bidId));
        }
        return resps;
    }

    /**
     * 执行监测信息
     */
    private void doTracker(TaskInfoRequest request, List<TaskInfo> resps, TaskCache taskCache, String bidId) {
        //任务请求埋点
        taskCounterIntegration.counterRequest(request, bidId);
        //任务响应埋点
        taskCounterIntegration.counterResponse(request, bidId, taskCache.getId());
    }

    /**
     * 应用是否有开启耍呗任务
     */
    private boolean checkApp(@RequestBody TaskInfoRequest request) {
        //变现猫appid(Code)
        App app = fetcher.fetch(AdxKeyGenerator.App.getInfo(request.getAppId()), App.class);
        if (null == app) {
            //客户appid(Code)
            KeyGenerator keyGenerator = AdxKeyGenerator.App.getInfoByCustom(request.getAppId(),
                    AppPlatformTypeEnum.convert(request.getOs()).getId());
            app = fetcher.fetch(keyGenerator, App.class);
            if (null == app) {
                log.warn("app is null, appId={}", request.getAppId());
                return false;
            }else{
                //将客户appid转换成变现猫appid
                request.setAppId(app.getAppId());
            }
        }
        if (!app.isOpenTask()) {
            log.debug("app not open task, appId={}", request.getAppId());
            return false;
        }
        return true;
    }

    /**
     * 自动填充参数
     */
    private void autoFillTaskInfoRequest(TaskInfoRequest request){
        //设备号
        String devUid = DevHelper.getUid(DevRequest.builder()
                .os(request.getOs())
                .imei(request.getImei())
                .imeiMd5(request.getImei_md5())
                .anid(request.getAnid())
                .anidMd5(request.getAnid_md5())
                .oaid(request.getOaid())
                .idfa(request.getIdfa())
                .idfaMd5(request.getIdfa_md5())
                .build());
        request.setDevUid(devUid);
        request.setIprst(ipService.analyze(request.getIp()));
        boolean osv_14 = null != request.getOs()
                && DevOsEnum.IOS.getId() == request.getOs()
                && StringUtils.isNotEmpty(request.getOsv())
                && StringHelper.isNewVersion("14", request.getOsv());
        request.setOsv_14(osv_14);
    }

    /**
     * 获取所有开放中的任务信息
     */
    private List<TaskCache> getOpenedList() {
        Map<String, TaskCache> taskCacheMap = fetcher.hfetchall(AdxKeyGenerator.Task.getOpenedList(),
                null, TaskCache.class);
        if (MapUtils.isEmpty(taskCacheMap)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(taskCacheMap.values());
    }

    /**
     * 封装任务信息
     */
    private TaskInfo getTaskInfo(TaskInfoRequest request, TaskCache taskCache, String bidId) {
        TaskInfo taskInfo = new TaskInfo();
        taskInfo.setTaskId(taskCache.getId());
        taskInfo.setTaskType(taskCache.getType());
        taskInfo.setWakePkn(taskCache.getPackageName());
        taskInfo.setWakeDeepUrl(taskCache.getJumpProtocol());
        taskInfo.setViewClkMax(taskCache.getViewClickInterval());
        if(null == taskCache.getWakeUpToDesktop() || taskCache.getWakeUpToDesktop() < 1){
            taskInfo.setWakeBackMax(3);//没有配置的情况下默认3秒
        }else{
            taskInfo.setWakeBackMax(taskCache.getWakeUpToDesktop());
        }
        taskInfo.setAcquireTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getAcquireMonitorUrl()));
        taskInfo.setViewTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getViewMonitorUrl()));
        taskInfo.setClickTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getClickMonitorUrl()));
        taskInfo.setReadyTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getReadyMonitorUrl()));
        taskInfo.setWakeTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getWakeUpMonitorUrl()));
        taskInfo.setSucceedTrackers(getTrackers(request, bidId, taskCache.getId(), taskCache.getWakeSucceedMonitorUrl()));
        taskInfo.setCmdText(taskCache.getCmdText());
        return taskInfo;
    }

    /**
     * 获取监测地址
     */
    private List<String> getTrackers(TaskInfoRequest request, String bidId, Long taskId, String monitorUrl){
        if(StringUtils.isBlank(monitorUrl)){
            return null;
        }
        String[] urlArr = monitorUrl.split(Constants.SPLIT_COMMA);
        List<String> urls = new ArrayList<>();
        for (String url : urlArr) {
            if(!pattern.matcher(url).find()){
                //不是变现猫的埋点地址
                continue;
            }
            if(url.indexOf("?") > -1){
                urls.add(StringUtils.join(url, "&bidid=", bidId, "&appid=", request.getAppId(), "&taskid=", taskId));
            }else{
                urls.add(StringUtils.join(url, "?bidid=", bidId, "&appid=", request.getAppId(), "&taskid=", taskId));
            }
        }
        return urls;
    }

    @Override
    public List<TaskCounter> getByIds(String idsStr) {
        if(StringUtils.isBlank(idsStr)){
            return null;
        }
        Date today = new Date();
        String dmy = DateHelper.format(today, DateHelper.PATTERN_STR8);
        String[] ids = idsStr.split(Constants.SPLIT_COMMA);
        JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
        Jedis jedis = null;
        Map<Long, Response<Map<String, String>>> mapRsMap = new HashMap<>();
        try {
            jedis = jedisPool.getResource();
            Pipeline pipeline = jedis.pipelined();
            for (String idStr : ids) {
                Long id = Long.valueOf(idStr);
                mapRsMap.put(id, pipeline.hgetAll(AdxKeyGenerator.Task.getCounter(id, dmy).generateKey()));
            }
            pipeline.syncAndReturnAll();
        } catch (Exception e){
          log.error("getByIds error: idsStr={},e={}", 
                  idsStr, ExceptionUtils.getStackTrace(e));
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }

        //解析数据
        return getTaskCounters(mapRsMap);
    }

    /**
     * 解析数据
     */
    private List<TaskCounter> getTaskCounters(Map<Long, Response<Map<String, String>>> mapRsMap) {
        List<TaskCounter> resps = new ArrayList<>();
        Set<Long> keySet = mapRsMap.keySet();
        for (Long id : keySet) {
            Response<Map<String, String>> mapRs = mapRsMap.get(id);
            Map<String, String> map = mapRs.get();//统计信息
            if(MapUtils.isEmpty(map)){
                continue;
            }
            TaskCounter counter = new TaskCounter();
            resps.add(counter);
            counter.setId(id);
            
            String acquires = map.get(TaskCounterField.ACQUIRES);
            counter.setTodayAcquires(StringUtils.isBlank(acquires) ? 0 : Long.valueOf(acquires));
            String views = map.get(TaskCounterField.VIEWS);
            counter.setTodayViews(StringUtils.isBlank(views) ? 0 : Long.valueOf(views));
            String clicks = map.get(TaskCounterField.CLICKS);
            counter.setTodayClicks(StringUtils.isBlank(clicks) ? 0 : Long.valueOf(clicks));
            String readyWakes = map.get(TaskCounterField.READY_WAKES);
            counter.setTodayReadyWakes(StringUtils.isBlank(readyWakes) ? 0 : Long.valueOf(readyWakes));
            String wakes = map.get(TaskCounterField.WAKES);
            counter.setTodayWakes(StringUtils.isBlank(wakes) ? 0 : Long.valueOf(wakes));
            String succeedWakes = map.get(TaskCounterField.SUCCEED_WAKES);
            counter.setTodaySucceedWakes(StringUtils.isBlank(succeedWakes) ? 0 : Long.valueOf(succeedWakes));
        }
        return resps;
    }
}
