package com.bxm.adxcounter.service.common.transfer;

import java.math.BigDecimal;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;

import com.bxm.adxcounter.facade.model.AdxCounterDTO;
import com.bxm.adxcounter.facade.model.TaskCounterDTO;
import com.bxm.adxcounter.integration.user.UserIntegration;
import com.bxm.adxcounter.service.model.endpoint.AdxEndpoint;
import com.bxm.user.facade.DevRequest;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;

import lombok.extern.slf4j.Slf4j;

/**
 * 转换器-ADX<br/>
 *
 * @author kerry.jiang
 * @date 2019/12/24 21:16
 */
@Slf4j
@Configuration
public class AdxConvert {
    
    private final KeyGenerator KEY_DSP = ()->KeyBuilder.build("ADX", "DSPS");
    
    @Autowired
    private Fetcher fetcher;
    @Autowired
    private SdkConvert sdkConvert;
    @Autowired
    private UserIntegration  userIntegration;

    /**
     * 赢价转厘
     */
    private final double WIN_LI = 10D;
    
    /**
     * 参数转换<br/>
     * ADX<br/>
     *
     * @param  endpoint
     * @param  counter
     */
    public void of(AdxEndpoint endpoint, AdxCounterDTO counter){
        endpoint.of(counter);
        endpoint.setBxmBidId(counter.getBidid());
        endpoint.setPositionId(counter.getTagid());
        endpoint.setDspId(counter.getDspid());
        endpoint.setCreateId(counter.getCreateid());
        endpoint.setWinPrice(getAdxPrice(counter));
        endpoint.setStatus(counter.getStatus());
        endpoint.setActivityId(counter.getActid());
        endpoint.setSceneType(counter.getScene());
        final DevRequest devRequest = DevRequest.builder()
                .anid(counter.getAnid())
                .anidMd5(counter.getAnid_md5())
                .idfa(counter.getIdfa())
                .idfaMd5(counter.getIdfa_md5())
                .imei(counter.getImei())
                .imeiMd5(counter.getImei_md5())
                .oaid(counter.getOaid())
                .build();
        final String uid = userIntegration.getUid(devRequest);
        endpoint.setDevUid(uid);
        endpoint.setXDown(counter.getX_down());
        endpoint.setYDown(counter.getY_down());
        endpoint.setXUp(counter.getX_up());
        endpoint.setYUp(counter.getY_up());
        endpoint.setReqpw(counter.getReqpw());
        endpoint.setReqph(counter.getReqph());
        endpoint.setActpw(counter.getActpw());
        endpoint.setActph(counter.getActph());
        endpoint.setViDuration(counter.getVi_duration());

        //填充SDK初始化参数
        sdkConvert.fillSdkInitInfo(endpoint, counter);
    }
    
    /**
     * 参数转换<br/>
     * ADX-耍呗任务<br/>
     *
     * @param  endpoint
     * @param  counter
     */
    public void of(AdxEndpoint endpoint, TaskCounterDTO counter){
        endpoint.of(counter);
        endpoint.setBxmBidId(counter.getBidid());
        endpoint.setPositionId(counter.getTagid());
        endpoint.setDspId(counter.getDspid());

        //填充SDK初始化参数
        sdkConvert.fillSdkInitInfo(endpoint, counter);
    }

    /**
     * 获取价格<br/>
     */
    private Double getAdxPrice(AdxCounterDTO counter) {
        if(StringUtils.isBlank(counter.getWin())){
            return null;
        }
        
        String win = null;
        if(NumberUtils.isNumber(counter.getWin())){
            win = counter.getWin();
        }else{
            //变现猫解密
            Dsp dsp = fetcher.hfetch(KEY_DSP, counter.getDspid(), Dsp.class);
            if(null == dsp) {
                log.error("Not find DSP, dspId={}", counter.getDspid());
                return null;
            }
            try {
                win = AdxDecrypt.decrypt(counter.getWin(), dsp.getApiKey());
            } catch (Exception e) {
                log.error("decrypt price fail, win={}", counter.getWin());
                return null;
            }
        }
        //分转厘
        return new BigDecimal(win).doubleValue() * WIN_LI;
    }
}
