package com.bxm.adxcounter.service.facade;

import javax.annotation.Resource;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.bxm.adx.facade.constant.enums.AppPlatformTypeEnum;
import com.bxm.adx.facade.constant.enums.DevOsEnum;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.model.dev.SdkInitCache;
import com.bxm.adx.facade.model.position.Position;
import com.bxm.adx.facade.rule.app.App;
import com.bxm.adxcounter.facade.constant.SdkMtEnum;
import com.bxm.adxcounter.facade.model.PositionVo;
import com.bxm.adxcounter.facade.model.SdkCounterDTO;
import com.bxm.adxcounter.facade.model.SdkEquipmentDTO;
import com.bxm.adxcounter.facade.model.SdkPublicDTO;
import com.bxm.adxcounter.facade.service.SdkService;
import com.bxm.adxcounter.integration.user.UserIntegration;
import com.bxm.adxcounter.service.common.Constant;
import com.bxm.adxcounter.service.common.transfer.SdkConvert;
import com.bxm.adxcounter.service.model.dev.InstalledApp;
import com.bxm.adxcounter.service.model.endpoint.SdkEndpoint;
import com.bxm.adxcounter.service.service.SdkEndpointService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;

import lombok.extern.slf4j.Slf4j;

@RestController
@Slf4j
public class SdkServiceImpl implements SdkService {

    @Autowired
    private SdkEndpointService sdkEndpointService;
    @Autowired
    private SdkConvert sdkConvert;
    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Resource
    private UserIntegration userIntegration;
    
    @Override
    public ResponseEntity<PositionVo> init(@RequestBody SdkEquipmentDTO sdk) {
        //自动填充参数
        autoFillSdkEquipmentDTO(sdk);

        //设备号
        String devNo = userIntegration.getUid(sdk);
        sdk.setDevUid(devNo);
        if(StringUtils.isNotEmpty(devNo)){
            //缓存SDK初始化信息
            cacheSdkInitInfo(devNo, sdk);
            //缓存已安装的app列表
            if (StringUtils.isNotBlank(sdk.getAppl())){
                List<String> packageList = new ArrayList<>();
                List<InstalledApp> appList = JSON.parseObject(sdk.getAppl(), new TypeReference<ArrayList<InstalledApp>>() {
                });
                if(CollectionUtils.isNotEmpty(appList)){
                    appList.forEach(app -> {
                        packageList.add(app.getPackageName());
                    });
                }
                KeyGenerator keyGenerator = AdxKeyGenerator.Device.getInitInstalledList(devNo);
                updater.update(keyGenerator, packageList, Constant.EXPIRE_SDK_INIT);
            }
        }
        sdkEndpointService.init(sdk);

        // 获取推送券的配置
        KeyGenerator keyGenerator = AdxKeyGenerator.App.getInfo(sdk.getAppid());
        App app = fetcher.fetch(keyGenerator, App.class);

        if (app == null || StringUtils.isBlank(app.getPushScenePositionId())){
            return ResponseEntity.ok().build();
        }

        Position position = fetcher.fetch(AdxKeyGenerator.Position.getInfo(app.getPushScenePositionId()), Position.class);
        if(null == position){
            return ResponseEntity.ok(null);
        }

        PositionVo positionVo = new PositionVo();

        positionVo.setPushTimesPerPersonPerDay(position.getPushTimesPerPersonPerDay());
        positionVo.setPushShowStayTime(position.getPushShowStayTime());
        positionVo.setPushIntervalTime(position.getPushIntervalTime());
        positionVo.setSdkInitIntervalTime(position.getSdkInitIntervalTime());
        positionVo.setAppId(app.getAppId());
        positionVo.setPositonId(app.getPushScenePositionId());
        return ResponseEntity.ok(positionVo);
    }

    /**
     * 自动填充参数
     */
    private void autoFillSdkEquipmentDTO(@RequestBody SdkEquipmentDTO sdk) {
        if(null == sdk.getTimestamp()){
            sdk.setTimestamp(System.currentTimeMillis());
        }
        if(StringUtils.isNotBlank(sdk.getAppid())){
            //客户appid(Code)
            DevOsEnum devOsEnum = DevOsEnum.of(sdk.getOs());
            KeyGenerator keyGenerator = AdxKeyGenerator.App.getInfoByCustom(sdk.getAppid(),
                    AppPlatformTypeEnum.convert(sdk.getOs()).getId());
            App app = fetcher.fetch(keyGenerator, App.class);
            if(null != app){
                //将客户appid转换成变现猫appid
                sdk.setAppid(app.getAppId());
            }
        }
    }

    /**
     * 缓存SDK初始化信息
     */
    private void cacheSdkInitInfo(String devNo, SdkEquipmentDTO sdk){
        SdkInitCache sdkCache = new SdkInitCache();
        sdkCache.setMac(sdk.getMac());
        sdkCache.setOs(sdk.getOs());
        sdkCache.setOsv(sdk.getOsv());
        sdkCache.setImei(sdk.getImei());
        sdkCache.setImei_md5(sdk.getImei_md5());
        sdkCache.setAnid(sdk.getAnid());
        sdkCache.setAnid_md5(sdk.getAnid_md5());
        sdkCache.setOaid(sdk.getOaid());
        sdkCache.setIdfa(sdk.getIdfa());
        sdkCache.setIdfa_md5(sdk.getIdfa_md5());
        sdkCache.setDevid(sdk.getDevid());
        sdkCache.setDevb(sdk.getDevb());
        sdkCache.setDevm(sdk.getDevm());
        sdkCache.setDevt(sdk.getDevt());
        sdkCache.setDevsh(sdk.getDevsh());
        sdkCache.setDevsw(sdk.getDevsw());
        sdkCache.setMemory(sdk.getMemory());
        sdkCache.setStorage(sdk.getStorage());
        sdkCache.setCpum(sdk.getCpum());
        sdkCache.setIsp(sdk.getIsp());
        sdkCache.setNet(sdk.getNet());
        sdkCache.setLon(sdk.getLon());
        sdkCache.setLat(sdk.getLat());
        sdkCache.setSdkv(sdk.getSdkv());
        sdkCache.setPkn(sdk.getPkn());

        KeyGenerator keyGenerator = AdxKeyGenerator.Device.getInitInfo(devNo);
        updater.update(keyGenerator, sdkCache, Constant.EXPIRE_SDK_INIT);
    }

    @Override
    public void extBind(@RequestBody SdkPublicDTO dto) {
        sdkEndpointService.extBind(dto);
    }

    @Override
    public ResponseEntity counter(@RequestBody SdkCounterDTO counter) {
        SdkMtEnum mtEnum = SdkMtEnum.of(counter.getMt());
        if(SdkMtEnum._OTHER == mtEnum){
            return ResponseEntity.badRequest().body("mt不匹配");
        }
        SdkEndpoint endpoint = new SdkEndpoint();
        sdkConvert.of(endpoint, counter);
        return sdkEndpointService.post(mtEnum, endpoint);
    }
}
