package com.bxm.adxcounter.service.service.impl;

import java.io.IOException;
import java.io.Serializable;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.commons.lang3.RandomUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.annotation.JSONField;
import com.bxm.adxcounter.facade.model.SdkEquipmentDTO;
import com.bxm.adxcounter.service.common.autoconfigure.config.ApplicationGlobalConfig;
import com.bxm.adxcounter.service.common.autoconfigure.config.GetuiConfig;
import com.bxm.adxcounter.service.service.GetuiService;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.http.HttpClientUtils;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * @author allen
 * @date 2020-11-11
 * @since 1.0
 */
@Slf4j
@Service
public class GetuiServiceImpl implements GetuiService, MeterBinder {

    private final ApplicationGlobalConfig config;
    private final Fetcher fetcher;
    private final JedisPool jedisPool;

    private CloseableHttpClient httpClient;
    private MeterRegistry registry;
    private Timer timer;
    private Map<String, Counter> failOnType = Maps.newConcurrentMap();

    public GetuiServiceImpl(ApplicationGlobalConfig config, Fetcher fetcher, JedisPool jedisPool) {
        this.config = config;
        this.fetcher = fetcher;
        this.jedisPool = jedisPool;

        final SocketConfig socketConfig = SocketConfig.custom().setTcpNoDelay(true).build();

        final PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(
                5000, TimeUnit.MILLISECONDS);
        connectionManager.setDefaultSocketConfig(socketConfig);
        connectionManager.setDefaultMaxPerRoute(3000 / 12);
        connectionManager.setMaxTotal(3000);
        //监测时间，毫秒
        connectionManager.setValidateAfterInactivity(5000);

        final RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(200)
                .setConnectTimeout(30)
                .setConnectionRequestTimeout(10)
                .setStaleConnectionCheckEnabled(true)
                .build();

        this.httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager)
                .build();
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
        this.timer = Timer.builder("getui.requests").register(registry);
    }

    @Override
    public List<String> fetchTags(SdkEquipmentDTO endpoint) {
        if(StringUtils.isEmpty(endpoint.getDevUid())){
            return null;
        }
        GetuiRequest request = ofEndpoint(endpoint);
        String key = KeyBuilder.build("tmp", "getui", "tags", endpoint.getDevUid());
        List<String> tags = fetcher.fetchToList(new TargetFactory<String>()
                .keyGenerator(() -> key)
                .skipNativeCache(true)
                .cls(String.class)
                .expireTimeInSecond(24 * 3600)
                .selector(5)
                .listDataExtractor(() -> fetchTags(request))
                .build());

        if (!CollectionUtils.isEmpty(tags)) {
            // Next week. TUESDAY 00:00:00
            long expireAt = LocalDateTime.of(LocalDate.now().plusDays(9 - LocalDate.now().getDayOfWeek().getValue()), LocalTime.of(0, 0)).toInstant(ZoneOffset.of("+8")).toEpochMilli() / 1000;
            try (Jedis jedis = jedisPool.getResource()){
                jedis.expireAt(key, expireAt + RandomUtils.nextLong(0, 30 * 60));
            }
        }
        return tags;
    }

    private GetuiRequest ofEndpoint(SdkEquipmentDTO endpoint) {
        String empty = "";
        return GetuiRequest.builder()
                .imeiMD5(StringUtils.isNotBlank(endpoint.getImei_md5()) ? endpoint.getImei_md5() : StringUtils.isNotBlank(endpoint.getImei()) ? DigestUtils.md5Hex(endpoint.getImei()) : empty)
                .macMD5(StringUtils.isNotBlank(endpoint.getMac()) ? DigestUtils.md5Hex(endpoint.getMac()) : empty)
                .imsiMD5(empty)
                .idfa(endpoint.getIdfa())
                .idfaMD5(StringUtils.isNotBlank(endpoint.getIdfa_md5()) ? endpoint.getIdfa_md5() : StringUtils.isNotBlank(endpoint.getIdfa()) ? DigestUtils.md5Hex(endpoint.getIdfa()) : empty)
                .oaid(endpoint.getOaid())
                .oaidMD5(StringUtils.isNotBlank(endpoint.getOaid()) ? DigestUtils.md5Hex(endpoint.getOaid()) : empty)
                .androidId(endpoint.getAnid())
                .androidIdMD5(StringUtils.isNotBlank(endpoint.getAnid_md5()) ? endpoint.getAnid_md5() : StringUtils.isNotBlank(endpoint.getAnid()) ? DigestUtils.md5Hex(endpoint.getAnid()) : empty)
                .build();
    }

    private List<String> fetchTags(GetuiRequest request) {
        long start = System.nanoTime();
        String url = config.getGetui().getFetchTagUrl();
        Map<String, String> headers = Maps.newHashMap();
        headers.put("Authorization", "Bearer " + getAccessCode());
        headers.put("Accept", "application/vnd.dmp.v1+json");

        try {
            String body = postRequestBody(url, request, headers);
            if (StringUtils.isNotBlank(body)) {
                GetuiResponse response = JsonHelper.convert(body, GetuiResponse.class);
                return response.getData();
            }
        } catch (Exception e) {
            String message = e.getClass().getSimpleName();
            Counter counter = failOnType.get(message);
            if (Objects.isNull(counter)) {
                counter = Counter.builder("getui.fail").tags("type", message).register(registry);
                failOnType.put(message, counter);
            }
            counter.increment();
        } finally {
            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
        return Lists.newArrayList();
    }

    private String getAccessCode() {
        return fetcher.fetch(new TargetFactory<String>()
                .keyGenerator(() -> KeyBuilder.build("tmp", "getui", "access_token"))
                .skipNativeCache(false)
                .cls(String.class)
                .expireTimeInSecond(7000)
                .selector(5)
                .dataExtractor(() -> {
                    GetuiConfig getui = config.getGetui();
                    String userCode = getui.getUserCode();
                    String authCode = getui.getAuthCode();
                    String authUrl = getui.getAuthUrl();

                    String timestamp = String.valueOf(System.currentTimeMillis());

                    Map<String, Object> params = Maps.newHashMap();
                    params.put("user_code", userCode);
                    params.put("sign1", DigestUtils.md5Hex(userCode + DigestUtils.md5Hex(timestamp)));
                    params.put("sign2", DigestUtils.md5Hex(userCode + authCode + DigestUtils.md5Hex(timestamp)));
                    params.put("timestamp", timestamp);

                    Map<String, String> headers = Maps.newHashMap();

                    try {
                        String body = get(authUrl, params, headers);
                        if (StringUtils.isNotBlank(body)) {
                            GetuiAccessResponse response = JsonHelper.convert(body, GetuiAccessResponse.class);
                            return response.getAccessToken();
                        }
                    } catch (Exception e) {
                        log.warn("getAccessCode: ", e);
                    }
                    return null;
                })
                .build());
    }

    private String get(String url, Map<String, Object> params, Map<String, String> headers)throws IOException  {
        HttpGet httpGet = new HttpGet(OkHttpUtils.appendParams(url, params));
        HttpClientUtils.setHeader(httpGet, headers);
        return doRequest(httpGet, headers);
    }

    private String postRequestBody(String url, Object o, Map<String, String> headers) throws IOException {
        HttpPost httpPost = new HttpPost(url);
        HttpClientUtils.setRequestBody(httpPost, o);
        return doRequest(httpPost, headers);
    }

    private String doRequest(HttpRequestBase httpReq, Map<String, String> headers) throws IOException {
        CloseableHttpResponse httpResp = null;
        try {
            HttpClientUtils.setHeader(httpReq, headers);
            httpResp = httpClient.execute(httpReq);
            return EntityUtils.toString(httpResp.getEntity());
        } finally {
            if (httpResp != null) {
                httpResp.close();
            }
        }
    }

    @Data
    @Builder
    @AllArgsConstructor
    private static class GetuiRequest implements Serializable {

        private static final long serialVersionUID = -4643729044689196299L;

        private String imeiMD5;
        private String macMD5;
        private String imsiMD5;
        private String mb;
        private String idfa;
        private String idfaMD5;
        private String oaid;
        private String oaidMD5;
        private String androidId;
        private String androidIdMD5;

        @Override
        public String toString() {
            return ToStringBuilder.reflectionToString(this, ToStringStyle.SIMPLE_STYLE);
        }
    }

    @Data
    private static class GetuiResponse implements Serializable {

        private static final long serialVersionUID = 8092303601506991826L;
        private String code;
        private List<String> data = Lists.newArrayList();
    }

    @Data
    private static class GetuiAccessResponse implements Serializable {

        private static final long serialVersionUID = -3405031987864169602L;
        private String code;
        @JSONField(name = "access_token")
        private String accessToken;
        @JSONField(name = "expires_in")
        private Long expiresIn;
    }
}
