package com.bxm.adxcounter.service.service.impl;

import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.adx.facade.model.position.Position;
import com.bxm.adxcounter.facade.model.UserData;
import com.bxm.adxcounter.service.service.UserService;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author allen
 * @date 2020-07-10
 * @since 1.0
 */
@Slf4j
@Service
public class UserServiceImpl implements UserService {

    private static final int EXPIRE_TIME = 7 * 24 * 60 * 60;

    private final Counter counter;
    private final Updater updater;
    private final Fetcher fetcher;

    public UserServiceImpl(Counter counter, Updater updater, Fetcher fetcher) {
        this.counter = counter;
        this.updater = updater;
        this.fetcher = fetcher;
    }

    @Override
    public void plusPayVideoFinished(String positionId, String uid) {
        Position position = getPosition(positionId);
        if (null == position) {
            log.info("Cannot found position by {}", positionId);
            return;
        }
        String appId = position.getAppId();
        String cuid = getCuid(appId, uid);
        if (StringUtils.isBlank(cuid)) {
            return;
        }
        // TODO 有可能因单个 hash 过大引起 Redis 分区数据倾斜而不稳定。
        counter.hincrementAndGet(getPayVideoFinishedKeyGenerator(appId), cuid, EXPIRE_TIME);
    }

    @Override
    public long getPayVideoFinishedByAppId(String appId, String cuid, String date) {
        if (StringUtils.isBlank(appId) || StringUtils.isBlank(cuid)) {
            return 0L;
        }
        KeyGenerator kg = getPayVideoFinishedKeyGenerator(date, appId);
        return counter.hget(kg, cuid);
    }

    @Override
    public List<UserData> getPayVideoFinishedByAppId(String appId, String[] cuids, String date) {
        if (ArrayUtils.isEmpty(cuids) || cuids.length > 999) {
            throw new IllegalArgumentException("cuids");
        }
        Object original = counter.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            return Collections.emptyList();
        }
        JedisPool jedisPool = (JedisPool) original;
        try (Jedis jedis = jedisPool.getResource()) {
            Map<String, Response<String>> result = Maps.newHashMap();
            Pipeline pipelined = jedis.pipelined();
            for (String cuid : cuids) {
                KeyGenerator k = getPayVideoFinishedKeyGenerator(date, appId);
                result.put(cuid, pipelined.hget(k.generateKey(), cuid));
            }
            pipelined.syncAndReturnAll();

            if (MapUtils.isNotEmpty(result)) {
                List<UserData> rs = Lists.newArrayListWithCapacity(result.size());
                for (Map.Entry<String, Response<String>> entry : result.entrySet()) {
                    String cuid = entry.getKey();
                    Response<String> redisRes = entry.getValue();
                    long times = NumberUtils.toLong(redisRes.get());
                    if (times > 0) {
                        rs.add(UserData.builder()
                                .cuid(cuid)
                                .finishedPlayVideoTimes(times)
                                .build());
                    }
                }
                return rs;
            }
        }
        return null;
    }

    @Override
    public void setUserMapping(String appId, String uid, String cuid) {
        updater.hupdateWithSelector(getUserMappingKeyGenerator(appId), uid, cuid, 1);
    }

    private String getCuid(String appId, String uid) {
        return fetcher.hfetchWithSelector(getUserMappingKeyGenerator(appId), uid, String.class, 1);
    }

    private Position getPosition(String positionId) {
        return fetcher.fetch(AdxKeyGenerator.Position.getInfo(positionId), Position.class);
    }

    private static KeyGenerator getPayVideoFinishedKeyGenerator(String appId) {
        return getPayVideoFinishedKeyGenerator(DateHelper.getDate(), appId);
    }

    private static KeyGenerator getPayVideoFinishedKeyGenerator(String date, String appId) {
        return () -> KeyBuilder.build("ADX", "COUNTER", "FINISHED_PLAY_VIDEO", date, appId);
    }

    private static KeyGenerator getUserMappingKeyGenerator(String appid) {
        return () -> KeyBuilder.build("ADX", "USERMAPPING", appid);
    }
}
